## ----eval=FALSE---------------------------------------------------------------
#  BiocManager::install("CBNplot")

## ----deg, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE, warning=FALSE, comment=FALSE, fig.height = 10, fig.width = 10----
library(CBNplot)
library(bnlearn)
library(DESeq2)
library(org.Hs.eg.db)
library(GEOquery)
## Load dataset and make metadata
filePaths <- getGEOSuppFiles("GSE133624")
counts = read.table(rownames(filePaths)[1], header=1, row.names=1)
meta = sapply(colnames(counts), function (x) substring(x,1,1))
meta = data.frame(meta)
colnames(meta) = c("Condition")

dds <- DESeqDataSetFromMatrix(countData = counts,
                              colData = meta,
                              design= ~ Condition)
## Prefiltering
filt <- rowSums(counts(dds) < 10) > dim(meta)[1]*0.9
dds <- dds[!filt,]

## Perform DESeq2()
dds = DESeq(dds)
res = results(dds, pAdjustMethod = "bonferroni")

## apply variance stabilizing transformation
v = vst(dds, blind=FALSE)
vsted = assay(v)

## Define the input genes, and use clusterProfiler::bitr to convert the ID.
sig = subset(res, padj<0.05)
cand.entrez = clusterProfiler::bitr(rownames(sig),
  fromType="ENSEMBL", toType="ENTREZID", OrgDb=org.Hs.eg.db)$ENTREZID

## Perform enrichment analysis
pway = ReactomePA::enrichPathway(gene = cand.entrez)
pway = clusterProfiler::setReadable(pway, org.Hs.eg.db)

## Define including samples
incSample = rownames(subset(meta, Condition=="T"))

## ----usecase, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE, warning=FALSE, comment=FALSE, fig.height = 10, fig.width = 10----
bngeneplot(results = pway,exp = vsted,
  expSample = incSample, pathNum = 15)

## ----usecase2, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE, warning=FALSE, comment=FALSE, fig.height = 10, fig.width = 10----
ret <- bngeneplot(results = pway,exp = vsted,
  expSample = incSample, pathNum = 15, returnNet=TRUE)
ret$str |> head()

## ----igraph, include=TRUE, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE----
g <- bnlearn::as.igraph(ret$av)
igraph::evcent(g)$vector

## ----usecase3, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE, warning=FALSE, comment=FALSE, fig.height = 10, fig.width = 10----
bnpathplot(results = pway,exp = vsted,
  expSample = incSample, nCategory=10, shadowText = TRUE)

## ----usecase4, include=TRUE, echo=TRUE, message=FALSE, cache=FALSE, warning=FALSE, comment=FALSE, fig.height = 10, fig.width = 10----
bnpathplotCustom(results = pway, exp = vsted, expSample = incSample,
  fontFamily="serif", glowEdgeNum=3, hub=3)
bngeneplotCustom(results = pway, exp = vsted, expSample = incSample,
  pathNum=15, fontFamily="sans", glowEdgeNum=NULL, hub=3)

## -----------------------------------------------------------------------------
sessionInfo()

