% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ProcessBAM_docs.R, R/ProcessBAM.R
\name{processBAM}
\alias{processBAM}
\alias{BAM2COV}
\title{Runs the OpenMP/C++ based SpliceWiz algorithm}
\usage{
BAM2COV(
  bamfiles = "./Unsorted.bam",
  sample_names = "sample1",
  output_path = "./cov_folder",
  n_threads = 1,
  useOpenMP = TRUE,
  overwrite = FALSE,
  verbose = FALSE,
  multiRead = FALSE
)

processBAM(
  bamfiles = "./Unsorted.bam",
  sample_names = "sample1",
  reference_path = "./Reference",
  output_path = "./SpliceWiz_Output",
  n_threads = 1,
  useOpenMP = TRUE,
  overwrite = FALSE,
  run_featureCounts = FALSE,
  verbose = FALSE,
  skipCOVfiles = FALSE,
  multiRead = FALSE
)
}
\arguments{
\item{bamfiles}{A vector containing file paths of 1 or more BAM files}

\item{sample_names}{The sample names of the given BAM files. Must
be a vector of the same length as \code{bamfiles}}

\item{output_path}{The output directory of this function}

\item{n_threads}{(default \code{1}) The number of threads to use. See details.}

\item{useOpenMP}{(default \code{TRUE}) Whether to use OpenMP.
If set to \code{FALSE}, BiocParallel will be used if \code{n_threads} is set}

\item{overwrite}{(default \code{FALSE}) If output files already exist,
will not attempt to re-run. If \code{run_featureCounts} is \code{TRUE}, will not
overwrite gene counts of previous run unless \code{overwrite} is \code{TRUE}.}

\item{verbose}{(default \code{FALSE}) Set to \code{TRUE} to allow \code{processBAM()} to
output progress bars and messages}

\item{multiRead}{(default \code{FALSE}) Whether SpliceWiz/ompBAM should use one
(set to \code{FALSE}) or all available threads (set to \code{TRUE}) to read BAM
files from the storage drive. In SSD drives or high performance computing
clusters, setting to \code{TRUE} may slightly improve performance, whereas if
reading from disk is the speed bottleneck, the default setting \code{FALSE}
should result in higher performance.}

\item{reference_path}{The directory containing the SpliceWiz reference}

\item{run_featureCounts}{(default \code{FALSE}) Whether this function will run
\link[Rsubread:featureCounts]{Rsubread::featureCounts} on the BAM files after counting spliced reads.
If so, the output will be
saved to \verb{"main.FC.Rds} in the \code{output_path} directory as a list object.}

\item{skipCOVfiles}{(default \code{FALSE}) Whether processBAM should skip
the production of COV files (containing coverage data). Default is to
create COV files unless this is set to \code{TRUE}. COV files can be generated
separately using \link{BAM2COV}}
}
\value{
Output will be saved to \code{output_path}. Output files
will be named using the given \code{sample_names}.
For \code{processBAM()}:
\itemize{
\item sample.txt.gz: The main output file containing the quantitation
of IR and splice junctions, as well as QC information\cr\cr
\item sample.cov: Contains coverage information in compressed binary. See
\link{getCoverage}
\item main.FC.Rds: A single file containing gene counts for the whole dataset
(only if \code{run_featureCounts == TRUE})
}

For \code{BAM2COV()}:
\itemize{
\item sample.cov: Contains coverage information in compressed binary. See
\link{getCoverage}
}
}
\description{
These function calls the SpliceWiz C++ routine on one or more BAM files.
\cr\cr The routine is an improved version over the original IRFinder, with
OpenMP-based multi-threading and the production of compact "COV" files to
record alignment coverage. A SpliceWiz reference built using
\link{Build-Reference-methods} is required.\cr\cr
After \code{processBAM()} is run, users should call
\link{collateData} to collate individual outputs into an experiment / dataset.
\cr\cr
BAM2COV creates COV files from BAM files without running \code{processBAM()}.
\cr\cr See details for performance info.
}
\details{
Typical run-times for a 100-million paired-end alignment BAM file takes 10
minutes using a single core. Using 8 threads, the runtime is approximately
2-5 minutes, depending on your system's file input / output speeds.
Approximately 10 Gb of RAM is used when OpenMP is used. If OpenMP
is not used (see below), this memory usage is multiplied across the number
of processor threads (i.e. 40 Gb if \code{n_threads = 4}).

OpenMP is natively available to Linux / Windows compilers, and OpenMP will
be used if \code{useOpenMP} is set to \code{TRUE}, using multiple threads to process
each BAM file. On Macs, if OpenMP is not available at compilation,
BiocParallel will be used, processing BAM files simultaneously,
with one BAM file per thread.
}
\section{Functions}{
\itemize{
\item \code{BAM2COV()}: Converts BAM files to COV files without running
\code{processBAM()}

\item \code{processBAM()}: Processes BAM files. Requires a
SpliceWiz reference generated by buildRef()

}}
\examples{

# Run BAM2COV, which only produces COV files but does not run `processBAM()`:

bams <- SpliceWiz_example_bams()

BAM2COV(bams$path, bams$sample,
  output_path = file.path(tempdir(), "SpliceWiz_Output"),
  n_threads = 2, overwrite = TRUE
)

# Run processBAM(), which produces:
# - text output of intron coverage and spliced read counts
# - COV files which record read coverages

example_ref <- file.path(tempdir(), "Reference")

buildRef(
    reference_path = example_ref,
    fasta = chrZ_genome(),
    gtf = chrZ_gtf()
)

bams <- SpliceWiz_example_bams()

processBAM(bams$path, bams$sample,
  reference_path = file.path(tempdir(), "Reference"),
  output_path = file.path(tempdir(), "SpliceWiz_Output"),
  n_threads = 2
)
}
\seealso{
\link{Build-Reference-methods} \link{collateData} \link{isCOV}
}
