## ----style, echo = FALSE, results = 'asis'------------------------------------
BiocStyle::markdown()

## ----'install_dev', eval = FALSE----------------------------------------------
# if (!require("devtools")) install.packages("devtools")
# remotes::install_github("MicTott/SpotSweeper")

## ----'install', eval = FALSE--------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#     install.packages("BiocManager")
# }
# 
# BiocManager::install("SpotSweeper")

## ----example_spe--------------------------------------------------------------
library(SpotSweeper)

# load  Maynard et al DLPFC daatset
spe <- STexampleData::Visium_humanDLPFC()

# show column data before SpotSweeper
colnames(colData(spe))

# drop out-of-tissue spots
spe <- spe[, spe$in_tissue == 1]

## ----example_scuttle----------------------------------------------------------
# change from gene id to gene names
rownames(spe) <- rowData(spe)$gene_name

# identifying the mitochondrial transcripts
is.mito <- rownames(spe)[grepl("^MT-", rownames(spe))]

# calculating QC metrics for each spot using scuttle
spe <- scuttle::addPerCellQCMetrics(spe, subsets = list(Mito = is.mito))
colnames(colData(spe))

## ----example_local_outliers---------------------------------------------------
# library size
spe <- localOutliers(spe,
    metric = "sum",
    direction = "lower",
    log = TRUE
)

# unique genes
spe <- localOutliers(spe,
    metric = "detected",
    direction = "lower",
    log = TRUE
)

# mitochondrial percent
spe <- localOutliers(spe,
    metric = "subsets_Mito_percent",
    direction = "higher",
    log = FALSE
)

## ----example_combine_local_outliers-------------------------------------------
# combine all outliers into "local_outliers" column
spe$local_outliers <- as.logical(spe$sum_outliers) |
    as.logical(spe$detected_outliers) |
    as.logical(spe$subsets_Mito_percent_outliers)

## ----local_outlier_plot-------------------------------------------------------
library(escheR)
library(ggpubr)

# library size
p1 <- plotQC(spe,
    metric = "sum_log",
    outliers = "sum_outliers", point_size = 1.1
) +
    ggtitle("Library Size")

# unique genes
p2 <- plotQC(spe,
    metric = "detected_log",
    outliers = "detected_outliers", point_size = 1.1
) +
    ggtitle("Unique Genes")

# mitochondrial percent
p3 <- plotQC(spe,
    metric = "subsets_Mito_percent",
    outliers = "subsets_Mito_percent_outliers", point_size = 1.1
) +
    ggtitle("Mitochondrial Percent")

# all local outliers
p4 <- plotQC(spe,
    metric = "sum_log",
    outliers = "local_outliers", point_size = 1.1, stroke = 0.75
) +
    ggtitle("All Local Outliers")

# plot
plot_list <- list(p1, p2, p3, p4)
ggarrange(
    plotlist = plot_list,
    ncol = 2, nrow = 2,
    common.legend = FALSE
)

## ----example_artifactRemoval--------------------------------------------------
# load in DLPFC sample with hangnail artifact
data(DLPFC_artifact)
spe <- DLPFC_artifact

# inspect colData before artifact detection
colnames(colData(spe))

## ----artifact_QC_plots--------------------------------------------------------
# library size
p1 <- plotQC(spe,
    metric = "sum_umi",
    outliers = NULL, point_size = 1.1
) +
    ggtitle("Library Size")

# unique genes
p2 <- plotQC(spe,
    metric = "sum_gene",
    outliers = NULL, point_size = 1.1
) +
    ggtitle("Unique Genes")

# mitochondrial percent
p3 <- plotQC(spe,
    metric = "expr_chrM_ratio",
    outliers = NULL, point_size = 1.1
) +
    ggtitle("Mitochondrial Percent")

# plot
plot_list <- list(p1, p2, p3)
ggarrange(
    plotlist = plot_list,
    ncol = 3, nrow = 1,
    common.legend = FALSE
)

## ----artifact_plot------------------------------------------------------------
# find artifacts using SpotSweeper
spe <- findArtifacts(spe,
    mito_percent = "expr_chrM_ratio",
    mito_sum = "expr_chrM",
    n_rings = 5,
    name = "artifact"
)

# check that "artifact" is now in colData
colnames(colData(spe))

## ----artifact_visualization---------------------------------------------------
plotQC(spe,
    metric = "expr_chrM_ratio",
    outliers = "artifact", point_size = 1.1
) +
    ggtitle("Hangnail artifact")

## -----------------------------------------------------------------------------
utils::sessionInfo()

