#' A theme function for tree-and-leaf igraph objects.
#'
#' This function sets attributes of a tree-and-leaf igraph object for
#' plotting in the RedeR app interface.
#' 
#' @param tal An igraph object generated by the \code{\link{TreeAndLeaf}} function.
#' @param theme An integer ranging from 1 to 5 with desired theme. \cr
#' Options: \cr
#' 1- A clean black and blue theme, for additional customizations. \cr
#' 2- Green palette theme. \cr
#' 3- Blue palette theme. \cr
#' 4- Purple palette theme. \cr
#' 5- Red palette theme. \cr
#' For detailed customization, see attributes in the 
#' \code{\link[RedeR]{addGraph}} method. \cr
#' 
#' @return An igraph object with attributes for RedeR application.
#'
#' @seealso \code{\link[RedeR]{addGraph}}
#' @seealso \code{\link{treeAndLeaf}}
#'
#' @examples
#' library(RedeR)
#' hc <- hclust(dist(USArrests), "ave")
#' tal <- treeAndLeaf(hc)
#' tal <- formatTree(tal, theme = 5)
#'
#' @importFrom igraph V E  
#' @export

formatTree <- function(tal, theme = 1){
    tal.checks(name = "tal", para = tal)
    tal.checks(name = "theme", para = theme)
    tal <- .setSizeTheme(tal)
    tal <- .setColorTheme(tal, theme)
    return(tal)
}
#-------------------------------------------------------------------------------
.setSizeTheme <- function(tal){
    sz <- sum(V(tal)$isLeaf)
    sz <- sqrt(sz)
    igraph::V(tal)$nodeFontSize <- ceiling(min(max(15,sz),75))
    igraph::V(tal)$nodeFontSize[!V(tal)$isLeaf] <- 1
    igraph::V(tal)$nodeSize <- ceiling(min(max(15,sz),100))
    igraph::V(tal)$nodeSize[!V(tal)$isLeaf] <- ceiling(min(max(5,sz/6),15))
    igraph::E(tal)$edgeWidth <- ceiling(min(max(15,sz/2),50))
    igraph::V(tal)$nodeLineWidth  <- ceiling(min(max(5,sz/2),20))
    return(tal)
}
#-------------------------------------------------------------------------------
.setColorTheme <- function(tal, theme){
    #Set node line color
    switch (theme,
            igraph::V(tal)$nodeLineColor <- "#000000",
            igraph::V(tal)$nodeLineColor <- "#379683",
            igraph::V(tal)$nodeLineColor <- "#55CFD1",
            igraph::V(tal)$nodeLineColor <- "#7F61A5",
            igraph::V(tal)$nodeLineColor <- "#D4A59A")
    #Set node font color
    switch (theme,
            igraph::V(tal)$nodeFontColor <- "#000000",
            igraph::V(tal)$nodeFontColor <- "#05386B",
            igraph::V(tal)$nodeFontColor <- "#1F2833",
            igraph::V(tal)$nodeFontColor <- "#5C2018",
            igraph::V(tal)$nodeFontColor <- "#5C2018")
    #Set the edge color
    switch (theme,
            igraph::E(tal)$edgeColor <- "#000000",
            igraph::E(tal)$edgeColor <- "#379683",
            igraph::E(tal)$edgeColor <- "#55CFD1",
            igraph::E(tal)$edgeColor <- "#7F61A5",
            igraph::E(tal)$edgeColor <- "#D4A59A")
    #Set the node colors
    switch (theme,
            igraph::V(tal)$nodeColor <- "#190061",
            igraph::V(tal)$nodeColor <- "#5CDB95",
            igraph::V(tal)$nodeColor <- "#45A29E",
            igraph::V(tal)$nodeColor <- "#44318D",
            igraph::V(tal)$nodeColor <- "#BC4639")
    return(tal)
}
