% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compareCCA.R, R/plot.compareCCAObject.R
\name{compareCCA}
\alias{compareCCA}
\alias{plot.compareCCAObject}
\title{Compare Canonical Correlation Analysis (CCA) between Query and Reference Data}
\usage{
compareCCA(
  query_data,
  reference_data,
  query_cell_type_col,
  ref_cell_type_col,
  pc_subset = 1:5,
  assay_name = "logcounts"
)

\method{plot}{compareCCAObject}(x, ...)
}
\arguments{
\item{query_data}{A \code{\linkS4class{SingleCellExperiment}} object containing numeric expression matrix for the query cells.}

\item{reference_data}{A \code{\linkS4class{SingleCellExperiment}} object containing numeric expression matrix for the reference cells.}

\item{query_cell_type_col}{The column name in the \code{colData} of \code{query_data} that identifies the cell types.}

\item{ref_cell_type_col}{The column name in the \code{colData} of \code{reference_data} that identifies the cell types.}

\item{pc_subset}{A numeric vector specifying the subset of principal components (PCs)
to compare. Default is the first five PCs.}

\item{assay_name}{Name of the assay on which to perform computations. Default is "logcounts".}

\item{x}{A list containing the output from the \code{compareCCA} function.
This list should include \code{cosine_similarity} and \code{correlations}.}

\item{...}{Additional arguments passed to the plotting function.}
}
\value{
A list containing the following elements:
\describe{
  \item{coef_ref}{Canonical coefficients for the reference dataset.}
  \item{coef_query}{Canonical coefficients for the query dataset.}
  \item{cosine_similarity}{Cosine similarity values for the canonical variables.}
  \item{correlations}{Canonical correlations between the reference and query datasets.}
}

The S3 plot method returns a \code{ggplot} object representing the scatter plot of cosine similarities of CCA coefficients and correlations.
}
\description{
This function performs Canonical Correlation Analysis (CCA) between two datasets (query and reference) after
performing PCA on each dataset. It projects the query data onto the PCA space of the reference data and then
computes the cosine similarity of the canonical correlation vectors between the two datasets.

The S3 plot method generates a visualization of the output from the `compareCCA` function.
The plot shows the cosine similarities of canonical correlation analysis (CCA) coefficients,
with point sizes representing the correlations.
}
\details{
The function performs the following steps:
1. Projects the query data onto the PCA space of the reference data using the specified number of principal components.
2. Downsamples the datasets to ensure an equal number of rows for CCA.
3. Performs CCA on the projected datasets.
4. Computes the cosine similarity between the canonical correlation vectors and extracts the canonical correlations.

The cosine similarity provides a measure of alignment between the canonical correlation vectors of the two datasets.
Higher values indicate greater similarity.

The S3 plot method converts the input list into a data frame suitable for plotting with \code{ggplot2}.
Each point in the scatter plot represents the cosine similarity of CCA coefficients, with the size of the point
indicating the correlation.
}
\examples{
# Load libraries
library(scran)
library(scater)

# Load data
data("reference_data")
data("query_data")

# Extract CD4 cells
ref_data_subset <- reference_data[, which(reference_data$expert_annotation == "CD4")]
query_data_subset <- query_data[, which(query_data$expert_annotation == "CD4")]

# Selecting highly variable genes (can be customized by the user)
ref_top_genes <- getTopHVGs(ref_data_subset, n = 500)
query_top_genes <- getTopHVGs(query_data_subset, n = 500)

# Intersect the gene symbols to obtain common genes
common_genes <- intersect(ref_top_genes, query_top_genes)
ref_data_subset <- ref_data_subset[common_genes,]
query_data_subset <- query_data_subset[common_genes,]

# Run PCA on datasets separately
ref_data_subset <- runPCA(ref_data_subset)
query_data_subset <- runPCA(query_data_subset)

# Compare CCA
cca_comparison <- compareCCA(query_data = query_data_subset,
                             reference_data = ref_data_subset,
                             query_cell_type_col = "expert_annotation",
                             ref_cell_type_col = "expert_annotation",
                             pc_subset = 1:5)

# Visualize output of CCA comparison
plot(cca_comparison)


}
\references{
Hotelling, H. (1936). "Relations between two sets of variates". *Biometrika*, 28(3/4), 321-377. doi:10.2307/2333955.
}
\seealso{
\code{\link{plot.compareCCAObject}}

\code{\link{compareCCA}}
}
\author{
Anthony Christidis, \email{anthony-alexander_christidis@hms.harvard.edu}
}
\keyword{internal}
