## ----knitr-options, echo = FALSE, message = FALSE, warning = FALSE------------
knitr::opts_chunk$set(collapse = TRUE, comment = "#>")
options(max.print = 30)

## ----install-bioc, eval = FALSE, include=TRUE---------------------------------
#  BiocManager::install("simPIC")

## ----quickstart---------------------------------------------------------------
# Load package
suppressPackageStartupMessages({
    library(simPIC)
})

# Load test data
set.seed(12)
counts <- readRDS(system.file("extdata", "test.rds", package = "simPIC"))

# Estimate parameters
est <- simPICestimate(counts)

# Simulate data using estimated parameters
sim <- simPICsimulate(est)

## ----pic, eval=FALSE, include=TRUE--------------------------------------------
#  pic_mat <- PIC_counting(
#      cells = cells,
#      fragment_tsv_gz_file_location = fragment_tsv_gz_file_location,
#      peak_sets = peak_sets
#  )

## ----simPICparams-------------------------------------------------------------
sim.params <- newsimPICcount()

## ----params-------------------------------------------------------------------
sim.params

## ----getParam-----------------------------------------------------------------
simPICget(sim.params, "nPeaks")

## ----setParam-----------------------------------------------------------------
sim.params <- setsimPICparameters(sim.params, nPeaks = 2000)
simPICget(sim.params, "nPeaks")

## ----getParams-setParams------------------------------------------------------
# Set multiple parameters at once (using a list)
sim.params <- setsimPICparameters(sim.params,
    update = list(nPeaks = 8000, nCells = 500)
)
# Extract multiple parameters as a list
params <- simPICgetparameters(
    sim.params,
    c("nPeaks", "nCells", "peak.mean.shape")
)
# Set multiple parameters at once (using additional arguments)
params <- setsimPICparameters(sim.params,
    lib.size.sdlog = 3.5, lib.size.meanlog = 9.07
)
params

## ----simPICestimate-----------------------------------------------------------
# Get the counts from test data
count <- readRDS(system.file("extdata", "test.rds", package = "simPIC"))

# Check that counts is a dgCMatrix
class(count)
typeof(count)

# Check the dimensions, each row is a peak, each column is a cell
dim(count)

# Show the first few entries
count[1:5, 1:5]

new <- newsimPICcount()
new <- simPICestimate(count)

## estimating using gamma distribution
## new <- simPICestimate(count, pm.distr = "gamma")

## ----simPICsimulate-----------------------------------------------------------
sim <- simPICsimulate(new, nCells = 1000)
sim

## simulating using gamma distribution
## sim <- simPICsimulate(new, nCells =1000, pm.distr = "gamma")

## ----SCE----------------------------------------------------------------------
# Access the counts
counts(sim)[1:5, 1:5]
# Information about peaks
head(rowData(sim))
# Information about cells
head(colData(sim))
# Peak by cell matrices
names(assays(sim))

## ----comparison---------------------------------------------------------------
sim1 <- simPICsimulate(nPeaks = 20000, nCells = 1000)
sim2 <- simPICsimulate(nPeaks = 20000, nCells = 1000)
comparison <- simPICcompare(list(real = sim1, simPIC = sim2))

names(comparison)
names(comparison$Plots)

## ----comparison-means---------------------------------------------------------
comparison$Plots$Means

## ----citation-----------------------------------------------------------------
citation("simPIC")

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

