## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

options(
  rmarkdown.html_vignette.check_title = FALSE
)


## -----------------------------------------------------------------------------
#' Perform superclustering on high-dimensional cytometry data.
#'
#' This function applies the silly, hypothetical clustering algorithm
#' "supercluster" to high-dimensional cytometry data using user-specified
#' input variables/cytometry measurements.
#'
#' @param tof_tibble A `tof_tbl` or `tibble`.
#'
#' @param cluster_cols Unquoted column names indicating which columns in
#' `tof_tibble` to use in computing the supercluster clusters.
#' Supports tidyselect helpers.
#'
#' @param num_kmeans_clusters An integer indicating how many clusters should be
#' used for the two k-means clustering steps.
#'
#' @param sep A string to use when splicing the 2 k-means clustering assignments
#' to one another.
#'
#' @param ... Optional additional parameters to pass to
#' \code{\link[tidytof]{tof_cluster_kmeans}}
#'
#' @return A tibble with one column named `.supercluster_cluster` containing
#' a character vector of length `nrow(tof_tibble)` indicating the id of the
#' supercluster cluster to which each cell (i.e. each row) in `tof_tibble` was
#' assigned.
#'
#' @importFrom dplyr tibble
#'
tof_cluster_supercluster <-
    function(tof_tibble, cluster_cols, num_kmeans_clusters = 10L, sep = "_", ...) {
        kmeans_1 <-
            tof_tibble |>
            tof_cluster_kmeans(
                cluster_cols = {{ cluster_cols }},
                num_clusters = num_kmeans_clusters,
                ...
            )

        kmeans_2 <-
            tof_tibble |>
            tof_cluster_kmeans(
                cluster_cols = {{ cluster_cols }},
                num_clusters = num_kmeans_clusters,
                ...
            )

        final_result <-
            dplyr::tibble(
                .supercluster_cluster =
                    paste(kmeans_1$.kmeans_cluster, kmeans_2$.kmeans_cluster, sep = sep)
            )

        return(final_result)
    }

## -----------------------------------------------------------------------------
sessionInfo()

