% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shuffle_sequences.R
\name{shuffle_sequences}
\alias{shuffle_sequences}
\title{Shuffle input sequences.}
\usage{
shuffle_sequences(sequences, k = 1, method = "euler", nthreads = 1,
  rng.seed = sample.int(10000, 1), window = FALSE, window.size = 0.1,
  window.overlap = 0.01)
}
\arguments{
\item{sequences}{\code{\link{XStringSet}} Set of sequences to shuffle. Works
with any set of characters.}

\item{k}{\code{numeric(1)} K-let size.}

\item{method}{\code{character(1)} One of \code{c('euler', 'markov', 'linear')}.
Only relevant if \code{k > 1}. See details.}

\item{nthreads}{\code{numeric(1)} Run \code{\link[=shuffle_sequences]{shuffle_sequences()}} in parallel with \code{nthreads}
threads. \code{nthreads = 0} uses all available threads.
Note that no speed up will occur for jobs with only a single sequence.}

\item{rng.seed}{\code{numeric(1)} Set random number generator seed. Since shuffling
can occur simultaneously in multiple threads using C++, it cannot communicate
with the regular \code{R} random number generator state and thus requires an
independent seed. Each individual sequence in an \code{\link{XStringSet}} object will be
given the following seed: \code{rng.seed * index}. The default is to pick a random
number as chosen by \code{\link[=sample]{sample()}}, which effectively is making \code{\link[=shuffle_sequences]{shuffle_sequences()}}
dependent on the R RNG state.}

\item{window}{\code{logical(1)} Shuffle sequences iteratively over windows instead
of all at once.}

\item{window.size}{\code{numeric(1)} Window size. Can be a fraction less than one, or
an integer representing the actual window size.}

\item{window.overlap}{\code{numeric(1)} Overlap between windows. Can be a fraction less
than one, or an integer representing the actual overlap size.}
}
\value{
\code{\link{XStringSet}} The input sequences will be returned with
identical names and lengths.
}
\description{
Given a set of input sequences, shuffle the letters within those
sequences with any k-let size.
}
\details{
\subsection{markov method}{

If \code{method = 'markov'}, then the Markov model is used to
generate sequences which will maintain (on average) the k-let
frequencies. Please note that this method is not a 'true' shuffling, and
for short sequences (e.g. <100bp) this can result in slightly more
dissimilar sequences versus true shuffling. See
Fitch (1983) for a discussion on the
topic.
}

\subsection{euler method}{

If \code{method = 'euler'}, then the sequence shuffling method proposed by
Altschul and Erickson (1985) is used. As opposed
to the 'markov' method, this one preserves exact k-let frequencies. This
is done by creating a k-let edge graph, then following a
random Eulerian walk through the graph. Not all walks will use up all
available letters however, so the cycle-popping algorithm proposed by
Propp and Wilson (1998) is used to find a
random Eulerian path. A side effect of using this method is that the
starting and ending sequence letters will remain unshuffled.
}

\subsection{linear method}{

If \code{method = 'linear'}, then the input sequences are split linearly
every \code{k} letters. For example, for \code{k = 3} 'ACAGATAGACCC' becomes
'ACA GAT AGA CCC'; after which these \code{3}-lets are shuffled randomly.
}

\subsection{Single-letter shuffling}{

Do note however, that the \code{method} parameter is only relevant for \code{k > 1}.
For \code{k = 1}, a simple shuffling is performed using the \code{shuffle} function
from the C++ standard library.
}
}
\examples{
if (R.Version()$arch != "i386") {
sequences <- create_sequences()
sequences.shuffled <- shuffle_sequences(sequences, k = 2)
}

}
\references{
Altschul SF, Erickson BW (1985). “Significance of Nucleotide
Sequence Alignments: A Method for Random Sequence Permutation That
Preserves Dinucleotide and Codon Usage.” \emph{Molecular Biology and
Evolution}, \strong{2}, 526-538.

Fitch WM (1983). “Random sequences.” \emph{Journal of Molecular
Biology}, \strong{163}, 171-176.

Propp JG, Wilson DW (1998). “How to get a perfectly random sample
from a generic markov chain and generate a random spanning tree of
a directed graph.” \emph{Journal of Algorithms}, \strong{27}, 170-217.
}
\seealso{
\code{\link[=create_sequences]{create_sequences()}}, \code{\link[=scan_sequences]{scan_sequences()}}, \code{\link[=enrich_motifs]{enrich_motifs()}},
\code{\link[=shuffle_motifs]{shuffle_motifs()}}
}
\author{
Benjamin Jean-Marie Tremblay, \email{benjamin.tremblay@uwaterloo.ca}
}
