% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/models.R
\docType{class}
\name{ModelComponent}
\alias{ModelComponent}
\title{An S4 class to represent model components}
\description{
\code{\link{ModelComponent}} objects bundle the actual mathematical
function for a particular component with various associated data
necesarry to incorporate them into a complete NLS model.
}
\details{
To be included in the automatic processing of potential model
components, a \code{\link{ModelComponent}} needs to be added to the
variable \code{fhComponents}.
}
\section{Slots}{

\describe{
\item{\code{name}}{character, a convenient name with which to refer to the
component}

\item{\code{desc}}{character, a short description of the component, for human
readers}

\item{\code{color}}{character, the color to use when plotting the component}

\item{\code{includeTest}}{function, a function which takes a single argument, a
\code{\link{FlowHist}} object, and returns \code{TRUE} if the
component should be included in the model for that object.}

\item{\code{function}}{function, a single-line function that returns the value
of the component. The function can take multiple arguments, which
usually will include \code{xx}, the bin number (i.e., x value) of the
histogram. The other arguments are model parameters, and should be
included in the \code{initParams} function.}

\item{\code{initParams}}{function, a function with a single argument, a
\code{\link{FlowHist}} object, which returns named list of model
parameters and their initial estimates.}

\item{\code{specialParams}}{list, a named list. The names are variables to
exclude from the default argument list, as they aren't parameters to
fit in the NLS procedure, but are actually fixed values. The body of
the list element is the object to insert into the model formula to
account for that variable. Note that this slot is not set directly,
but should be provided by the value returned by
\code{specialParamSetter} (which by default is \code{list(xx =
substitute(xx))}).}

\item{\code{specialParamSetter}}{function, a function with one argument, the
\code{\link{FlowHist}} object, used to set the value of
\code{specialParams}. This allows parameters to be declared 'special'
based on values in the \code{\link{FlowHist}} object. The default
value for this slot is a function which returns \code{list(xx =
substitute(xx))}}

\item{\code{paramLimits}}{list, a named list with the upper and lower limits of
each parameter in the function.}

\item{\code{doCounts}}{logical, should cell counts be evaluated for this
component? Used to exclude the debris models, which don't work with
R's Integrate function.}
}}

\section{Coding Concepts}{


See the source code file \code{models.R} for the actual code used in
defining model components. Here are a few examples to illustrate
different concepts.

We'll start with the G1 peaks. They are modelled by the components
\code{fA1} and \code{fB1} (for the A and B samples). The
\code{includeTest} for \code{fA1} is simply \code{function(fh) TRUE},
since there will always be at least one peak to fit. \code{fB1} is
included if there is more than 1 detected peak, and the setting
\code{samples} is more than 1, so the \code{includeTest} is
\preformatted{function(fh) nrow(fhPeaks(fh)) > 1 && fhSamples(fh) > 1}

The G1 component is defined by the function
\preformatted{(a1 / (sqrt(2 * pi) * Sa) * exp(-((xx - Ma)^2)/(2 *
  Sa^2)))} 

with the arguments \code{a1, Ma, Sa, xx}. \code{xx} is treated
specially, by default, and we don't need to deal with it here. The
initial estimates for the other parameters are calculated in
\code{initParams}:
\preformatted{function(fh){
  Ma <- as.numeric(fhPeaks(fh)[1, "mean"])
  Sa <- as.numeric(Ma / 20)
  a1 <- as.numeric(fhPeaks(fh)[1, "height"] * Sa / 0.45)
  list(Ma = Ma, Sa = Sa, a1 = a1)
}
}

\code{Ma} is the mean of the distribution, which should be very close to
the peak. \code{Sa} is the standard distribution of the distribution.
If we assume the CV is 5\%, that means the \code{Sa} should be 5\% of
the distribution mean, which gives us a good first estimate.
\code{a1} is a scaling parameter, and I came up with the initial
estimate by trial-and-error. Given the other two values are going to be
reasonably close, the starting value of \code{a1} doesn't seem to be
that crucial.

The limits for these values are provided in \code{paramLimits}.
\preformatted{paramLimits = list(Ma = c(0, Inf), Sa = c(0, Inf), a1 =
  c(0, Inf))}

They're all bound between 0 and Infinity. The upper bound for \code{Ma}
and \code{Sa} could be lowered to the number of bins, but I haven't had
time or need to explore this yet.

The G2 peaks include the \code{d} argument, which is the ratio of the G2
peak to the G1 peak. That is, the linearity parameter:
\preformatted{func = function(a2, Ma, Sa, d, xx){
  (a2 / (sqrt(2 * pi) * Sa * 2) *
    exp(-((xx - Ma * d)^2)/(2 * (Sa * 2)^2))) 
}
}

\code{d} is the ratio between the G2 and G1 peaks. If \code{linearity =
"fixed"}, it is set to 2. Otherwise, it is fit as a model parameter.
This requires special handling. First, we check the \code{linearity}
value in \code{initParams}, and provide a value for \code{d} if needed:
\preformatted{res <- list(a2 = a2)
if(fhLinearity(fh) == "variable")
    res <- c(res, d = 2)
}

Here, \code{a2} is always treated as a parameter, and \code{d} is
appended to the initial paramter list only if needed.

We also need to use the \code{specialParamSetter} function, in this case
calling the helper function \code{setLinearity(fh)}. This function
checks the value of \code{linearity}, and returns the appropriate object
depending on the result.

Note that we use the arguments \code{Ma} and \code{Sa} appear in the
\code{function} slot for \code{fA2}, but we don't need to provide their
initial values or limits. These values are already supplied in the
definition of \code{fA1}, which is always present when \code{fA2} is.

NB.: This isn't checked in the code! I know \code{fA1} is always
present, but there is no automated checking of this fact. If you create
a \code{ModelComponent} that has parameters that are not defined in that
component, and are not defined in other components (like \code{Ma} is in
this case), you will cause problems. There is also nothing to stop you
from defining a parameter multiple times. That is, you could define
initial estimates and limits for \code{Ma} in \code{fA1} and \code{fA2}.
This may also cause problems. It would be nice to do some
sanity-checking to protect against using parameters without defining
initial estimates or limits, or providing multiple/conflicting
definitions.

The Single-Cut Debris component is unusual in two ways. It doesn't
include the argument \code{xx}, but it uses the pre-computed values
\code{SCvals}. Consequently, we must provide a function for
\code{specialParamSetter} to deal with this:
\preformatted{specialParamSetter = function(fh){ list(SCvals =
substitute(SCvals)) } }

The Multi-Cut Debris component \code{MC} is similar, but it needs to
include \code{xx} as a special parameter. The aggregate component
\code{AG} also includes several special parameters.

For more discussion of the debris components, see
\code{\link{DebrisModels}}. 

The code responsible for this is in the file \code{models.R}. Accessor
functions are provided (but not exported) for getting and setting
\code{\link{ModelComponent}} slots. These functions are named
  \code{mcSLOT}, and include \code{mcFunc}, \code{mcColor},
  \code{mcName}, \code{mcDesc}, \code{mcSpecialParams},
  \code{mcSpecialParamSetter}, \code{mcIncludeTest},
  \code{mcInitParams}.
}

\examples{
## The 'master list' of components is stored in fhComponents:
flowPloidy:::fhComponents ## outputs a list of component summaries

## adding a new component to the list:
\dontrun{
fhComponents$pois <-
  new("ModelComponent", name = "pois", color = "bisque",
      desc = "A poisson component, as a silly example",
      includeTest = function(fh){
          ## in this case, we check for a flag in the opt slot
          ## We could also base the test on some feature of the
          ## data, perhaps something in the peaks or histData slots
          "pois" \%in\% fh@opt
      },
      func = function(xx, plam){
          ## The function needs to be complete on a single line, as it
          ## will be 'stitched' together with other functions to make
          ## the complete model.
          exp(-plam)*plam^xx/factorial(xx)
      },
      initParams = function(fh){
          ## If we were to use this function for one of our peaks, we
          ## could use the peak position as our initial estimate of
          ## the Poisson rate parameter:
          plam <- as.numeric(fhPeaks(fh)[1, "mean"])
      },
      ## bound the search for plam between 0 and infinity. Tighter
      ## bounds might be useful, if possible, in speeding up model
      ## fitting and avoiding local minima in extremes.
      paramLimits = list(plam = c(0, Inf)) 
  )

  ## specialParamSetter is not needed here - it will default to a
  ## function that returns "xx = xx", indicating that all other
  ## parameters will be fit. That is what we need for this example. If
  ## the component doesn't include xx, or includes other fixed
  ## parameters, then specialParamSetter will need to be provided.  

  ## Note that if our intention is to replace an existing component with
  ## a new one, we either need to explicitly change the includeTest for
  ## the existing component to account for situations when the new one
  ## is used instead. As a temporary hack, you could add both and then
  ## manually remove one with \code{dropComponents}. 
  }
}
