#' PAC_analyze
#' 
#' Given a PAC object and targets, this wrapper will perform several analyses of 
#' the PAC object and return a PDF with plots. More specifically, the plot
#' will contain a jitterplot of fold changes, stack bar of mean counts/cpm,
#' size distribution stack bars, piecharts, PCA plots, and if  specified, a 
#' volcano plot based off adjusted p-values and fold changes from a DESeq2 analysis. 
#'
#' @param PAC PAC-object containing Anno and a summary list-folder with
#'   summarized dataframe(s) for example generated by PAC_summary. The function
#'   can also be applied on a dataframe containing a column with annotation
#'   groupings (e.g. biotype) and a column with summarized data (e.g. log2 fold
#'   changes). Rows should be unique sequences as the rownames of a PAC summary
#'   object.
#'   
#' @param pheno_target List with: 1st object being a character vector
#'   of target column in Pheno and 2nd object being a character vector of the target
#'   group(s) in the target Pheno column (1st object).
#' 
#' @param norm Character indicating what type of data to be used. If 'counts',
#'   the raw counts in Counts will be used (default). If "cpm", cpm values will 
#'   be used. As of now, only "counts" and "cpm" may be used for this wrapper.
#' 
#' @param anno_target Character vector with the name of the target column in
#'   Anno or the name of the annotation column in case of input being a
#'   dataframe.
#' 
#' @param model Character of model used to run \code{PAC_deseq}.
#'
#' @return a list of plots in R, and a pdf file in home directory called
#' "Results_seqpac.pdf".
#'
#' @examples
#' 
#' load(system.file("extdata", "drosophila_sRNA_pac_filt_anno.Rdata", 
#'                  package = "seqpac", mustWork = TRUE))
#'
#'                  
#' result_list <- PAC_analyze(pac, 
#'   pheno_target=list("stage"),
#'   norm="cpm", 
#'   anno_target=list("Biotypes_mis0"),
#'   model=~stage+batch)
#' 
#' @export

PAC_analyze <- function(PAC, pheno_target=NULL, norm=NULL,
                        anno_target=NULL, model=NULL){
  
  res_list <- list(NA)
  
  if(norm=="cpm"){
    PAC <- PAC_summary(PAC, pheno_target = pheno_target, norm="cpm")
    PAC <- PAC_summary(PAC, pheno_target = pheno_target, norm="cpm", type="log2FC")
    jitter <- PAC_jitter(PAC, summary_target = list(paste0("Log2FC_",pheno_target[[1]])),
                         anno_target = anno_target)
    sb1<-PAC_stackbar(PAC, anno_target = anno_target, summary_target = list(paste0("cpmMeans_",pheno_target[[1]])), norm = "cpm")
    sb2<-PAC_stackbar(PAC, anno_target = anno_target, summary_target = list(paste0("cpmMeans_",pheno_target[[1]])), norm = "cpm")
    sd <- PAC_sizedist(PAC, norm="cpm", anno_target = anno_target, nucleotide_range =c(15,75),
                        summary_target = list(paste0("cpmMeans_",pheno_target[[1]])))
    # pie <- PAC_pie(PAC, pheno_target = pheno_target, anno_target=anno_target,
    #                summary="pheno")
    pca <- PAC_pca(PAC, pheno_target = pheno_target)
  }
  else{
    PAC <- PAC_summary(PAC, pheno_target = pheno_target, norm="counts")
    PAC <- PAC_summary(PAC, pheno_target = pheno_target, norm="counts", type="log2FC")
    jitter <- PAC_jitter(PAC, summary_target = list(paste0("Log2FC_",pheno_target[[1]])),
                         anno_target = anno_target)
    sb1<-PAC_stackbar(PAC, anno_target = anno_target, pheno_target = pheno_target, norm = "counts")
    sb2<-PAC_stackbar(PAC, anno_target = anno_target, pheno_target = pheno_target, norm = "counts",
                      summary_target= list(paste0("countsMeans_",pheno_target[[1]])))
    sd <- PAC_sizedist(PAC, norm="counts", anno_target = anno_target, nucleotide_range =c(15,75),
                       summary_target = list(paste0("countsMeans_",pheno_target[[1]])))
    # pie <- PAC_pie(PAC, pheno_target = pheno_target, anno_target=anno_target,
    #                summary="pheno")
    pca <- PAC_pca(PAC, pheno_target = pheno_target)
  }
  
  if(!is.null(model)){
  dsq <- PAC_deseq(PAC, model=model)
  }
  else{
  dsq <- PAC_deseq(PAC, model=pheno_target[[1]])
  }
  
  #Print all results in a pdf
  res_list <- c(jitter, list(sb1), list(sb2), sd$Histograms,
                list(pie),
                 pca$graphs, list(dsq$plots$volcano))
  
 
    pdf("Results_Seqpac.pdf", width = 7, height = 5)
    for (p in res_list) {
      print(p)
  }
  dev.off()
  
  return(res_list)
}
