% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/differential-expression.R
\name{deAnalysis}
\alias{deAnalysis}
\alias{performMirnaDE}
\alias{performGeneDE}
\title{Perform differential expression analysis}
\usage{
performMirnaDE(
  mirnaObj,
  group,
  contrast,
  design,
  method = "edgeR",
  logFC = 0,
  pCutoff = 0.05,
  pAdjustment = "fdr",
  filterByExpr.args = list(),
  calcNormFactors.args = list(),
  estimateDisp.args = list(robust = TRUE),
  glmQLFit.args = list(),
  glmQLFTest.args = list(),
  DESeq.args = list(),
  useVoomWithQualityWeights = TRUE,
  voom.args = list(),
  lmFit.args = list(),
  eBayes.args = list(),
  useArrayWeights = TRUE,
  useWsva = FALSE,
  wsva.args = list(),
  arrayWeights.args = list(),
  useDuplicateCorrelation = FALSE,
  correlationBlockVariable = NULL,
  duplicateCorrelation.args = list()
)

performGeneDE(
  mirnaObj,
  group,
  contrast,
  design,
  method = "edgeR",
  logFC = 0,
  pCutoff = 0.05,
  pAdjustment = "fdr",
  filterByExpr.args = list(),
  calcNormFactors.args = list(),
  estimateDisp.args = list(robust = TRUE),
  glmQLFit.args = list(),
  glmQLFTest.args = list(),
  DESeq.args = list(),
  useVoomWithQualityWeights = TRUE,
  voom.args = list(),
  lmFit.args = list(),
  eBayes.args = list(),
  useArrayWeights = TRUE,
  useWsva = FALSE,
  wsva.args = list(),
  arrayWeights.args = list(),
  useDuplicateCorrelation = FALSE,
  correlationBlockVariable = NULL,
  duplicateCorrelation.args = list()
)
}
\arguments{
\item{mirnaObj}{A \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object
containing miRNA and gene data}

\item{group}{The variable of interest for differential expression analysis.
It must be the column name of a variable present in the metadata (colData)
of a \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object. See the \emph{details}
section for additional information}

\item{contrast}{A \code{character} object that specifies the groups to be
compared during differential expression analysis, separated by a dash
(e.g. 'disease-healthy'). Note that reference group must be the last one,
for additional information see the \emph{details} section}

\item{design}{An R \code{formula} that indicates the model to fit. It must
include the variable of interest (\code{group}) together with eventual
covariates (e.g. '~ 0 + disease + sex'). Please note that \code{group} variable
must be the first one. See the \emph{details} section for additional information}

\item{method}{The statistical package used to compute differential
expression. For NGS experiments, it must be one of \code{edgeR} (default),
\code{DESeq2}, and \code{voom} (for limma-voom). Instead, for microarray data, only
\code{limma} can be used}

\item{logFC}{The minimum log2 fold change required to consider a gene as
differentially expressed. Optional, default is 0}

\item{pCutoff}{The adjusted p-value cutoff to use for statistical
significance. The default value is \code{0.05}}

\item{pAdjustment}{The p-value correction method for multiple testing. It
must be one of: \code{fdr} (default), \code{BH}, \code{none}, \code{holm}, \code{hochberg}, \code{hommel},
\code{bonferroni}, \code{BY}}

\item{filterByExpr.args}{A \code{list} object containing additional arguments
passed to \code{\link[edgeR:filterByExpr]{edgeR::filterByExpr()}} function. It is used when \code{method} is set
to \code{edgeR} or \code{voom}}

\item{calcNormFactors.args}{A \code{list} object containing additional arguments
passed to \code{\link[edgeR:calcNormFactors]{edgeR::calcNormFactors()}} function. It is used when \code{method} is
set to \code{edgeR} or \code{voom}}

\item{estimateDisp.args}{A \code{list} object containing additional arguments
passed to \code{\link[edgeR:estimateDisp]{edgeR::estimateDisp()}} function. It is used when \code{method} is
set to \code{edgeR}. Default is \code{list(robust = TRUE)} to use the robust parameter}

\item{glmQLFit.args}{A \code{list} object containing additional arguments
passed to \code{\link[edgeR:glmQLFit]{edgeR::glmQLFit()}} function. It is used when \code{method} is
set to \code{edgeR}}

\item{glmQLFTest.args}{A \code{list} object containing additional arguments
passed to \code{\link[edgeR:glmQLFTest]{edgeR::glmQLFTest()}} function. It is used when \code{method} is
set to \code{edgeR}}

\item{DESeq.args}{A \code{list} object containing additional arguments
passed to \code{\link[DESeq2:DESeq]{DESeq2::DESeq()}} function. It is used when \code{method} is
set to \code{DESeq}}

\item{useVoomWithQualityWeights}{Logical, whether to use the
\code{\link[limma:voomWithQualityWeights]{limma::voomWithQualityWeights()}} function or just the \code{\link[limma:voom]{limma::voom()}}
function. It is used when \code{method} is set to \code{voom}. Default is TRUE}

\item{voom.args}{A \code{list} object containing additional arguments
passed to \code{\link[limma:voom]{limma::voom()}} function or \code{\link[limma:voomWithQualityWeights]{limma::voomWithQualityWeights()}}
function. It is used when \code{method} is set to \code{voom}}

\item{lmFit.args}{A \code{list} object containing additional arguments
passed to \code{\link[limma:lmFit]{limma::lmFit()}} function. It is used when \code{method} is set
to \code{voom} or \code{limma}}

\item{eBayes.args}{A \code{list} object containing additional arguments
passed to \code{\link[limma:ebayes]{limma::eBayes()}} function. It is used when \code{method} is set
to \code{voom} or \code{limma}}

\item{useArrayWeights}{Logical, whether to use the \code{\link[limma:arrayWeights]{limma::arrayWeights()}}
function or not. It is used when \code{method} is set to \code{limma}. Default is TRUE}

\item{useWsva}{Logical, whether to use the \code{\link[limma:wsva]{limma::wsva()}} function or not.
It is used when \code{method} is set to \code{limma}. Default is FALSE}

\item{wsva.args}{A \code{list} object containing additional arguments
passed to \code{\link[limma:wsva]{limma::wsva()}} function. It is used when \code{method} is set
to \code{limma}}

\item{arrayWeights.args}{A \code{list} object containing additional arguments
passed to \code{\link[limma:arrayWeights]{limma::arrayWeights()}} function. It is used when \code{method} is set
to \code{limma}}

\item{useDuplicateCorrelation}{Logical, whether to use the
\code{\link[limma:dupcor]{limma::duplicateCorrelation()}} function or not. It is used when \code{method} is
set to \code{limma}. Default is FALSE}

\item{correlationBlockVariable}{It is the blocking variable to use for
\code{\link[limma:dupcor]{limma::duplicateCorrelation()}}. Default is NULL}

\item{duplicateCorrelation.args}{A \code{list} object containing additional
arguments passed to \code{\link[limma:dupcor]{limma::duplicateCorrelation()}} function. It is used
when \code{method} is set to \code{limma}}
}
\value{
A \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object containing differential
expression results. To access these results, the user may run the
\code{\link[=mirnaDE]{mirnaDE()}} and \code{\link[=geneDE]{geneDE()}} functions for miRNAs and genes, respectively.
}
\description{
\code{performMirnaDE()} and \code{performGeneDE()} are two functions provided by MIRit
to conduct miRNA and gene differential expression analysis, respectively.
In particular, these functions allow the user to compute differential
expression through different methods, namely \code{edgeR} (Quasi-Likelihood
framework), \code{DESeq2}, \code{limma-voom} and \code{limma}. Data deriving from NGS
experiments and microarray technology are all suitable for these functions.
For precise indications about how to use these functions, please refer to
the \emph{details} section.
}
\details{
When performing differential expression for NGS experiments, count matrices
are detected and \code{method} parameter must be one of \code{edgeR}, \code{DESeq2},
and \code{voom}. On the other hand, when dealing with microarray studies, only
\code{limma} can be used.

To calculate differential expression, MIRit must be informed about the
variable of interest and the desired contrast. In particular, the \code{group}
parameter must be the name of a variable present in the metadata (colData)
of a \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object, which specifies the
variable used to compute differential expression analysis, between the groups
indicated in \code{contrast}. Specifically, \code{contrast} must be a character vector
that defines the levels to compare separated by a dash. For example, if we
have a variable named 'condition', with two levels, namely 'disease' and
'healthy', we can identify differentially expressed genes in 'disease'
samples compared to 'healthy' subjects by specifying: \code{group = 'condition'}
and \code{contrast = 'disease-healthy'}. Furthermore, the user needs to specify
the model to fit expression values. To do so, the user has to state the
model formula in the \code{design} parameter. Please note that for a correct
inner working of these functions, the \code{group} variable of interest must be
the \emph{first} variable in model formula. Moreover, the user can include in the
design any other sources of variation by specifying covariates that will be
taken into account. For instance, if we want to compare 'disease' subjects
against 'healthy' individuals, without the influence of sex differences,
we may specify \code{design = ~ condition + sex}, where 'sex' is also a
variable present in the metadata (colData) of \code{mirnaObj}.

Notably, for all the methods available, the user can supply additional
arguments to the functions implemented in \code{edgeR}, \code{DESeq2} and \code{limma}.
Therefore, the user has finer control over how the differential expression
analysis is performed. In this regard, for microarray studies, the user
may opt to include weighted surrogate variable analysis (WSVA) to correct
for unknown sources of variation (\code{useWsva = TRUE}). Moreover, for
microarray data, the \code{arrayWeights()} function in \code{limma} can be used to
assess differential expression with respect to array qualities. Also, the
\code{duplicateCorrelation()} function in \code{limma} may be included in the pipeline
in order to block the effect of correlated samples. To do this, the user
must set \code{useDuplicateCorrelation = TRUE}, and must specify the blocking
variable through the \code{correlationBlockVariable} parameter. Additionally,
when using \code{limma-voom}, the user may estimate voom transformation with or
without quality weights (by specifying \code{useVoomWithQualityWeights = TRUE}).
}
\section{Functions}{
\itemize{
\item \code{performMirnaDE()}: Perform differential expression analysis for miRNAs

\item \code{performGeneDE()}: Perform differential expression analysis for genes

}}
\examples{
\donttest{
# load example MirnaExperiment object
obj <- loadExamples()

# perform miRNA DE with edgeR
obj <- performMirnaDE(obj,
    group = "disease", contrast = "PTC-NTH",
    design = ~ 0 + disease + patient, method = "edgeR"
)

# perform miRNA DE with DESeq2
obj <- performMirnaDE(obj,
    group = "disease", contrast = "PTC-NTH",
    design = ~ 0 + disease + patient, method = "DESeq2"
)

# perform miRNA DE with limma-voom
obj <- performMirnaDE(obj,
    group = "disease", contrast = "PTC-NTH",
    design = ~ 0 + disease + patient, method = "voom"
)
}

}
\references{
Ritchie ME, Phipson B, Wu D, Hu Y, Law CW, Shi W, Smyth GK (2015). “limma
powers differential expression analyses for RNA-sequencing and microarray
studies.” Nucleic Acids Research, 43(7), e47. \url{doi:10.1093/nar/gkv007}.

Law, CW, Chen, Y, Shi, W, and Smyth, GK (2014). "Voom: precision weights
unlock linear model analysis tools for RNA-seq read counts". Genome Biology
15, R29

Robinson MD, McCarthy DJ, Smyth GK (2010). “edgeR: a Bioconductor package
for differential expression analysis of digital gene expression data.”
Bioinformatics, 26(1), 139-140. \url{doi:10.1093/bioinformatics/btp616}.

Love MI, Huber W, Anders S (2014). “Moderated estimation of fold change and
dispersion for RNA-seq data with DESeq2.” Genome Biology, 15, 550.
\url{doi:10.1186/s13059-014-0550-8}.
}
\author{
Jacopo Ronchi, \email{jacopo.ronchi@unimib.it}
}
