% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregate.R, R/arithmetics.R, R/boost.R,
%   R/coarsen.R, R/normalize.R, R/subsample.R
\name{arithmetics}
\alias{arithmetics}
\alias{detrend}
\alias{autocorrelate}
\alias{divide}
\alias{merge,HiCExperiment,HiCExperiment-method}
\alias{despeckle}
\alias{aggregate,HiCExperiment-method}
\alias{boost}
\alias{subsample}
\alias{coarsen}
\alias{normalize,HiCExperiment-method}
\title{HiContacts arithmetics functionalities}
\usage{
\S4method{aggregate}{HiCExperiment}(
  x,
  targets,
  flankingBins = 51,
  maxDistance = NULL,
  BPPARAM = BiocParallel::bpparam()
)

detrend(x, use.scores = "balanced")

autocorrelate(x, use.scores = "balanced", detrend = TRUE, ignore_ndiags = 3)

divide(x, by, use.scores = "balanced", pseudocount = 0)

\S4method{merge}{HiCExperiment,HiCExperiment}(x, y, ..., use.scores = "balanced", FUN = mean)

despeckle(x, use.scores = "balanced", focal.size = 1)

boost(x, use.scores = "balanced", alpha = 1, full.replace = FALSE)

coarsen(x, bin.size)

\S4method{normalize}{HiCExperiment}(
  object,
  use.scores = "count",
  niters = 200,
  min.nnz = 10,
  mad.max = 3
)

subsample(x, prop)
}
\arguments{
\item{x, y, object}{a \code{HiCExperiment} object}

\item{targets}{Set of chromosome coordinates for which
interaction counts are extracted from the Hi-C contact file, provided
as a GRanges object (for diagnoal-centered loci) or as a GInteractions
object (for off-diagonal coordinates).}

\item{flankingBins}{Number of bins on each flank of the bins containing
input targets.}

\item{maxDistance}{Maximum distance to use when compiling distance decay}

\item{BPPARAM}{BiocParallel parameters}

\item{use.scores}{Which scores to use to perform operations}

\item{detrend}{Detrend matrix before performing autocorrelation}

\item{ignore_ndiags}{ignore N diagonals when calculating correlations}

\item{by}{a \code{HiCExperiment} object}

\item{pseudocount}{Add a pseudocount when dividing matrices (Default: 0)}

\item{...}{\code{HiCExperiment} objects. For \code{aggregate}, \code{targets} (a set of
GRanges or GInteractions).}

\item{FUN}{merging function}

\item{focal.size}{Size of the smoothing rectangle}

\item{alpha}{Power law scaling factor. As indicated in Boost-HiC documentation,
the alpha parameter influences the weighting of contacts: if alpha < 1
long-range interactions are prioritized; if alpha >> 1 short-range
interactions have more weight when computing the distance matrix.}

\item{full.replace}{Whether to replace the entire set of contacts,
rather than only filling the missing interactions (count=0) (Default: FALSE)}

\item{bin.size}{Bin size to coarsen a HiCExperiment at}

\item{niters}{Number of iterations for ICE matrix balancing}

\item{min.nnz}{Filter bins with less than \code{min.nnz} non-zero elements when
performing ICE matrix balancing}

\item{mad.max}{Filter out bins whose log coverage is less than \code{mad.max}
median absolute deviations below the median bin log coverage.}

\item{prop}{Float between 0 and 1, or integer corresponding to the # of}
}
\value{
a \code{HiCExperiment} object with extra scores
}
\description{
Different arithmetic operations can be performed on Hi-C contact matrices:
\itemize{
\item \code{normalize} a contact matrix using iterative correction;
\item \code{detrend} a contact matrix, i.e. remove the distance-dependent
contact trend;
\item \code{autocorrelate} a contact matrix: this is is typically done to highlight
large-scale compartments;
\item \code{divide} one contact matrix by another;
\item \code{merge} multiple contact matrices;
\item \code{despeckle} (i.e. smooth out) a contact matrix out;
\item \code{aggregate} (average) a contact matrices over a set of genomic loci of
interest;
\item \code{boost} Hi-C signal by enhancing long-range interactions while preserving short-
range interactions (this is adapted from Boost-HiC);
\item \code{subsample} interactions using a proportion or a fixed number of final
interactions.
\item \code{coarsen} a contact matrix to a larger (coarser) resolution
}
}
\examples{
#### -----
#### Normalize a contact matrix
#### -----

library(HiContacts)
contacts_yeast <- contacts_yeast()
normalize(contacts_yeast)

#### -----
#### Detrending a contact matrix
#### -----

detrend(contacts_yeast)

#### -----
#### Auto-correlate a contact matrix
#### -----

autocorrelate(contacts_yeast)

#### -----
#### Divide 2 contact matrices
#### -----

contacts_yeast <- refocus(contacts_yeast, 'II')
contacts_yeast_eco1 <- contacts_yeast_eco1() |> refocus('II')
divide(contacts_yeast_eco1, by = contacts_yeast)

#### -----
#### Merge 2 contact matrices
#### -----

merge(contacts_yeast_eco1, contacts_yeast)

#### -----
#### Despeckle (smoothen) a contact map
#### -----

despeckle(contacts_yeast)

#### -----
#### Aggregate a contact matrix over centromeres, at different scales
#### -----

contacts <- contacts_yeast() |> zoom(resolution = 1000)
centros <- topologicalFeatures(contacts, 'centromeres')
aggregate(contacts, targets = centros, flankingBins = 51)

#### -----
#### Enhance long-range interaction signal
#### -----

contacts <- contacts_yeast() |> zoom(resolution = 1000) |> refocus('II')
boost(contacts, alpha = 1)

#### -----
#### Subsample & "coarsen" contact matrix 
#### -----

subcontacts <- subsample(contacts, prop = 100000)
coarsened_subcontacts <- coarsen(subcontacts, bin.size = 4000)
}
