#' @title plotPhyloTree
#' 
#' @param phyloTree phyloTree or phyloTreeList object generated by \code{\link{getPhyloTree}} function.
#' @param patient.id Select the specific patients. Default NULL, all patients are included.
#' @param branchCol Specify the colors of branches Default 'mutType'. Other options: "mutSig" for coloring branches by branch mutation signature;
#' @param show.bootstrap Logical (Default: TRUE). Whether to add bootstrap value on internal nodes. 
#' @param min.ratio Double. Default 1/20. If min.ratio is not NULL,
#'  all edge length of a phylogenetic tree should be greater than
#'  min.ratio*the longest edge length.
#'  If not, the edge length will be reset as min.ratio*longest edge length.
#' @param signaturesRef Signature reference,Users can upload their own reference. Default "cosmic_v2". Option:"exome_cosmic_v3","nature2013".
#' @param min.mut.count The threshold for the variants in a branch. Default 15.
#' @param use.tumorSampleLabel Logical (Default: FALSE). Rename the 'Tumor_Sample_Barcode' with 'Tumor_Sample_Label'.
#' @param show.scale.bar Logical (Default: FALSE). Whether to show scale bar.This function adds a horizontal bar giving the scale of the branch lengths to a plot on the current graphical device.
#' @param scale.bar.x The x location of scale bar.
#' @param scale.bar.y The y location of scale bar.
#' @examples
#' maf.File <- system.file("extdata/", "CRC_HZ.maf", package = "MesKit")
#' clin.File <- system.file("extdata/", "CRC_HZ.clin.txt", package = "MesKit")
#' ccf.File <- system.file("extdata/", "CRC_HZ.ccf.tsv", package = "MesKit")
#' maf <- readMaf(mafFile=maf.File, clinicalFile = clin.File, ccfFile=ccf.File, refBuild="hg19")
#' 
#' phyloTree <- getPhyloTree(maf, patient.id = 'V402')
#' plotPhyloTree(phyloTree)
#
#' @return return a list of phylotree graph .
#' @import ape ggplot2 ggrepel
#' @export plotPhyloTree
#' 


plotPhyloTree <- function(phyloTree,
                          patient.id = NULL,
                          branchCol = "mutType",
                          show.bootstrap = TRUE,
                          min.ratio = 1/20,
                          signaturesRef = "cosmic_v2",
                          min.mut.count = 15,
                          use.tumorSampleLabel = FALSE,
                          show.scale.bar = FALSE,
                          scale.bar.x = NULL,
                          scale.bar.y = NULL){
    
    processPlotTree <- function(phyloTree){
       patient <- getPhyloTreePatient(phyloTree)
       p <- plotTree(phyloTree,
                     min.ratio = min.ratio,
                     branchCol = branchCol,
                     show.bootstrap = show.bootstrap,
                     signaturesRef = signaturesRef,
                     min.mut.count = min.mut.count,
                     use.tumorSampleLabel = use.tumorSampleLabel,
                     show.scale.bar = show.scale.bar,
                     scale.bar.x = scale.bar.x,
                     scale.bar.y = scale.bar.y)
       return(p)
    }
    
    if(!is.null(branchCol)){
       branchCol.options <- c("mutSig","mutType")
       if(!branchCol %in% branchCol.options){
          stop("'branchCol' should be NULL, 'mutType' or 'mutSig'.")
       }
    }
    
    
   phylotree_input <- subPhyloTree(phyloTree,patient.id = patient.id)
    
   tree_list <- lapply(phylotree_input, processPlotTree)
   tree_list <- tree_list[!is.na(tree_list)]
   
   if(length(tree_list)==1){
       return(tree_list[[1]])
   }else if(length(tree_list) == 0){
       return(NA)
   }else{
       return(tree_list)
   }
}