% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interaction_model.R
\name{interaction_model}
\alias{interaction_model}
\title{Fits linear models with interaction to triplet data (Target, TF, DNAm), where DNAm
is a binary variable (samples in Q1 or Q4)}
\usage{
interaction_model(
  triplet,
  dnam,
  exp,
  dnam.group.threshold = 0.25,
  cores = 1,
  tf.activity.es = NULL,
  sig.threshold = 0.05,
  fdr = TRUE,
  filter.correlated.tf.exp.dnam = TRUE,
  filter.correlated.target.exp.dnam = TRUE,
  filter.triplet.by.sig.term = TRUE,
  stage.wise.analysis = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{triplet}{Data frame with columns for DNA methylation region (regionID), TF  (TF), and target gene  (target)}

\item{dnam}{DNA methylation matrix or SummarizedExperiment object
(columns: samples in the same order as \code{exp} matrix, rows: regions/probes)}

\item{exp}{A matrix or SummarizedExperiment object object
 (columns: samples in the same order as \code{dnam},
rows: genes represented by ensembl IDs (e.g. ENSG00000239415))}

\item{dnam.group.threshold}{DNA methylation threshold percentage to define samples 
in the low methylated group and high methylated group. For example, 
setting the threshold to 0.3 (30\%) will assign samples with the lowest 30\% 
methylation in the low group and the highest 30\% methylation in the high group. 
Default is 0.25 (25\%), accepted threshold range (0.0,0.5].}

\item{cores}{Number of CPU cores to be used. Default 1.}

\item{tf.activity.es}{A matrix with normalized enrichment scores for each TF across all samples
to be used in linear models instead of TF gene expression. See \code{\link{get_tf_ES}}.}

\item{sig.threshold}{Threshold to filter significant triplets.
Select if interaction.pval < 0.05 or pval.dnam < 0.05 or pval.tf < 0.05 in binary model}

\item{fdr}{Uses fdr when using sig.threshold.
Select if interaction.fdr < 0.05 or fdr.dnam < 0.05 or fdr.tf < 0.05 in binary model}

\item{filter.correlated.tf.exp.dnam}{If wilcoxon test of TF expression Q1 and Q4 is significant (pvalue < 0.05),
triplet will be removed.}

\item{filter.correlated.target.exp.dnam}{If wilcoxon test of target expression Q1 and Q4 is not significant (pvalue > 0.05),
triplet will be removed.}

\item{filter.triplet.by.sig.term}{Filter significant triplets ?
Select if interaction.pval < 0.05 or pval.dnam <0.05 or pval.tf < 0.05 in binary model}

\item{stage.wise.analysis}{A boolean indicating if stagewise analysis should be performed
to correct for multiple comparisons. If set to FALSE FDR analysis is performed.}

\item{verbose}{A logical argument indicating if
messages output should be provided.}
}
\value{
A dataframe with \code{Region, TF, target, TF_symbo, target_symbol, estimates and P-values},
after fitting robust linear models or zero-inflated negative binomial models (see Details above).

Model considering DNAm values as a binary variable generates \code{quant_pval_metGrp},
\code{quant_pval_rna.tf}, \code{quant_pval_metGrp.rna.tf},
\code{quant_estimates_metGrp}, \code{quant_estimates_rna.tf}, \code{quant_estimates_metGrp.rna.tf}.

\code{Model.interaction} indicates which model (robust linear model or zero inflated model)
was used to fit Model 1, and \code{Model.quantile} indicates which model(robust linear model or zero
inflated model) was used to fit Model 2.
}
\description{
Evaluates regulatory potential of DNA methylation (DNAm) on gene expression,
by fitting robust linear model or zero inflated negative binomial model to triplet data.
These models consist of terms to model direct effect of DNAm on target gene expression,
direct effect of TF on gene expression, as well as an interaction term that evaluates
the synergistic effect of DNAm and TF on gene expression.
}
\details{
This function fits the linear model

\code{log2(RNA target) ~ log2(TF) + DNAm + log2(TF) * DNAm}

to triplet data as follow:

Model by considering \code{DNAm} as a binary variable - we defined a binary group for
DNA methylation values (high = 1, low = 0). That is, samples with the highest
DNAm levels (top 25 percent) has high = 1, samples with lowest
DNAm levels (bottom 25 percent) has high = 0. Note that in this
implementation, only samples with DNAm values in the first and last quartiles
are considered.

In these models, the term \code{log2(TF)} evaluates direct effect of TF on
target gene expression, \code{DNAm} evaluates direct effect of DNAm on target
gene expression, and \code{log2(TF)*DNAm} evaluates synergistic effect of DNAm
and TF, that is, if TF regulatory activity is modified by DNAm.

There are two implementations of these models, depending on whether there are an excessive
amount (i.e. more than 25 percent) of samples with zero counts in RNAseq data:

\itemize{
\item When percent of zeros in RNAseq data is less than
25 percent, robust linear models are implemented using \code{rlm} function from \code{MASS} package. This
gives outlier gene expression values reduced weight. We used \code{"psi.bisqure"}
option in function \code{rlm} (bisquare weighting,
https://stats.idre.ucla.edu/r/dae/robust-regression/).

\item When percent of zeros in RNAseq data is more than 25 percent, zero inflated negative binomial models
are implemented using \code{zeroinfl} function from \code{pscl} package. This assumes there are
two processes that generated zeros (1) one where the counts are always zero
(2) another where the count follows a negative binomial distribution.
}

To account for confounding effects from covariate variables, first use the \code{get_residuals} function to obtain
RNA or DNAm residual values which have covariate effects removed, then fit interaction model. Note that no
log2 transformation is needed when \code{interaction_model} is applied to residuals data.

Note that only triplets with TF expression not significantly different in high vs. low
methylation groups will be evaluated (Wilcoxon test, p > 0.05).
}
\examples{
library(dplyr)
dnam <- runif(20,min = 0,max = 1) \%>\%
  matrix(ncol = 1) \%>\%  t
rownames(dnam) <- c("chr3:203727581-203728580")
colnames(dnam) <- paste0("Samples",1:20)

exp.target <-  runif(20,min = 0,max = 10) \%>\%
  matrix(ncol = 1) \%>\%  t
rownames(exp.target) <- c("ENSG00000252982")
colnames(exp.target) <- paste0("Samples",1:20)

exp.tf <- runif(20,min = 0,max = 10) \%>\%
  matrix(ncol = 1) \%>\%  t
rownames(exp.tf) <- c("ENSG00000083937")
colnames(exp.tf) <- paste0("Samples",1:20)

exp <- rbind(exp.tf, exp.target)

triplet <- data.frame(
   "regionID" =  c("chr3:203727581-203728580"),
   "target" = "ENSG00000252982",
   "TF" = "ENSG00000083937"
)
results <- interaction_model(
   triplet = triplet, 
   dnam = dnam, 
   exp = exp, 
    dnam.group.threshold = 0.25,
   stage.wise.analysis = FALSE, 
   sig.threshold = 1,
   filter.correlated.tf.exp.dnam = FALSE,
   filter.correlated.target.exp.dnam = FALSE,
   filter.triplet.by.sig.term = FALSE
)
}
