#' Computes various node centralities
#'
#' @description Computes degree, eigenvector centrality and betweenness
#' centrality for the ceRNA interaction network induced by the results of the
#' SPONGE method
#'
#' @param sponge_result output of the sponge method
#' @param directed Whether to consider the input network as directed or not.
#' @importFrom igraph graph.data.frame
#' @importFrom igraph eigen_centrality
#' @importFrom igraph betweenness
#' @importFrom igraph degree
#' @importFrom data.table data.table
#'
#' @return data table or data frame with gene, degree, eigenvector and betweenness
#' @export
#'
#' @seealso sponge
#'
#' @examples
#' sponge_node_centralities(ceRNA_interactions)
sponge_node_centralities <- function(sponge_result, directed = FALSE){

    network <- igraph::graph.data.frame(sponge_result)

    ev_centrality <- igraph::eigen_centrality(network,
                                              directed = directed)
    btw_centrality <- igraph::betweenness(network, directed = directed)

    page_rank <- igraph::page_rank(network, directed = directed)

    centrality_df <- data.table(gene = names(ev_centrality$vector),
                                degree = igraph::degree(network),
                                eigenvector = ev_centrality$vector,
                                betweenness = btw_centrality,
                                page_rank = page_rank$vector)
    return(centrality_df)
}

#' Computes edge centralities
#'
#' @description Computes edge betweenness
#' centrality for the ceRNA interaction network induced by the results of the
#' SPONGE method.
#'
#' @param sponge_result The output generated by the sponge method.
#'
#' @importFrom igraph graph.data.frame
#' @importFrom igraph E
#' @importFrom igraph edge_betweenness
#' @importFrom igraph degree
#'
#' @return data table or data frame with gene, degree, eigenvector and betweenness
#' @export
#'
#' @seealso sponge
#'
#' @examples
#'sponge_edge_centralities(ceRNA_interactions)
sponge_edge_centralities <- function(sponge_result){
    directed <- FALSE

    network <- igraph::graph.data.frame(sponge_result)

    edge_labels <- attr(E(network), "vnames")
    ebtw <- edge_betweenness(network, directed = directed)
    data.frame(sponge_result[,c("geneA", "geneB")], edge_betweenness = ebtw)
}

#' plot node network centralities
#'
#' @param network_centralities a result from sponge_node_centralities()
#' @param measure one of 'all', 'degree', 'ev' or 'btw'
#' @param x plot against another column in the data table, defaults to degree
#' @param top label the top x samples in the plot
#' @param base_size size of the text in the plot
#'
#' @return a plot
#' @export
#'
#' @examples \dontrun{
#' network_centralities <- sponge_node_centralities(ceRNA_interactions)
#' sponge_plot_network_centralities(network_centralities)}
sponge_plot_network_centralities <- function(network_centralities,
                                             measure="all",
                                             x = "degree",
                                             top = 5,
                                             base_size = 18){
    if(!requireNamespace("ggplot2", quietly = TRUE)){
        stop("Install the package ggplot2 for producing this plot")
    }
    if(!requireNamespace("ggrepel", quietly = TRUE)){
        warning("Install the package ggrepel for non-overlapping labels")
        geom_label_repel <- ggplot2::geom_label
    }else{
        geom_label_repel <- ggrepel::geom_label_repel
    }
    if(!requireNamespace("digest", quietly = TRUE)){
        warning("Install the package digest for better coloring")
        network_centralities$color <- "#000000"
    }else{
        network_centralities$color <-
            paste("#", substr(sapply(network_centralities$gene, function(x)
                digest::digest(x, algo = "crc32")), 1, 6), sep="")
    }

    p1 <- ggplot2::ggplot(network_centralities, ggplot2::aes(x=degree)) +
        ggplot2::geom_histogram(color=I("black"), fill=I("black"), alpha = 0.3)+
        ggplot2::theme_bw(base_size = base_size) +
        ggplot2::theme(strip.background = ggplot2::element_rect(fill="grey"))
    p2 <- ggplot2::ggplot(network_centralities, ggplot2::aes_string(x = x,
                                                  y = "eigenvector",
                                                  color= "color",
                                                  label = "gene")) +
        ggplot2::geom_point(alpha = 0.3) +
        ggplot2::ylab("eigenvector") +
        ggplot2::theme_bw(base_size = base_size) +
        ggplot2::theme(legend.position = "none") +
        geom_label_repel(data =
                             head(network_centralities[
                                 order(-network_centralities$eigenvector),
                                 ], top))
    p3 <- ggplot2::ggplot(network_centralities, ggplot2::aes_string(x = x,
                                                  y = "betweenness",
                                                  color = "color",
                                                  label = "gene")) +
        ggplot2::geom_point(alpha = 0.3) +
        ggplot2::ylab("betweenness") +
        ggplot2::theme_bw(base_size = base_size) +
        ggplot2::theme(legend.position = "none") +
        geom_label_repel(data =
                             head(network_centralities[
                                 order(-network_centralities$betweenness),
                                 ], top))

    if(measure == "degree") return(p1)
    else if(measure == "ev") return(p2)
    else if(measure == "btw") return(p3)
    else{
        if(!requireNamespace("gridExtra", quietly = TRUE)){
            warning("You need to install package gridExtra to combine plots")
            return(p1)
        }
        else{
        gridExtra::grid.arrange(p1,
                      p2 + ggplot2::theme(
                          strip.background = ggplot2::element_blank(),
                          strip.text.x = ggplot2::element_blank(),
                          legend.position = "none"),
                      p3 + ggplot2::theme(
                          strip.background = ggplot2::element_blank(),
                          strip.text.x = ggplot2::element_blank(),
                          legend.position = "none"),
                      ncol = 1)
        }
    }
}

