% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cell_distance.R
\name{calculateSanityDistance}
\alias{calculateSanityDistance}
\title{Calculate the Sanity distance between samples}
\usage{
calculateSanityDistance(
  x,
  assay = "logcounts",
  assay.sd = "logcounts_sd",
  gene_sd = "sanity_activity_sd",
  gene_mu = "sanity_log_activity_mean",
  mu_sd = "sanity_log_activity_mean_sd",
  snr_cutoff = 1,
  nbin = 400L,
  subset.row = NULL,
  BPPARAM = bpparam()
)
}
\arguments{
\item{x}{A \linkS4class{SingleCellExperiment} or \linkS4class{SummarizedExperiment}
object which stores the results of the Sanity analysis.}

\item{assay}{The name of the assay containing the log normalized counts matrix.}

\item{assay.sd}{The name of the assay containing the standard deviation of the log-normalized counts}

\item{gene_sd}{The name of the column in the \code{rowData(x)} that contains the standard deviation of the gene log-fold change.}

\item{gene_mu}{The name of the column in the \code{rowData(x)} that contains the mean log activity of the genes.}

\item{mu_sd}{The name of the column in the \code{rowData(x)} that contains the standard deviation of the mean log activity of the genes.}

\item{snr_cutoff}{A numeric value indicating the minimum signal-to-noise ratio (SNR) to consider a gene.}

\item{nbin}{Number of bins to use when calculating prior variance of the true distance.}

\item{subset.row}{A vector of row indices or logical vector indicating which rows to use.}

\item{BPPARAM}{A BiocParallelParam object specifying the parallelization strategy.}
}
\value{
A \linkS4class{dist} object containing the expected pairwise distances between cells.
}
\description{
Calculates the expected squared Euclidean distance between two cells using a
hierarchical model that shrinks noisy gene differences toward zero.
}
\details{
\subsection{Distance Calculation}{

The method calculates the expected squared Euclidean distance between two cells,
adjusting for uncertainty in gene expression estimates. For each gene \eqn{g},
the contribution to the squared distance between cells \eqn{c} and \eqn{c'} is:

\deqn{\langle \Delta_g^2 \rangle = x_g^2 f_g^2(\alpha) + \eta_g^2 f_g(\alpha)}

where:
\itemize{
\item \eqn{x_g = \delta_{gc} - \delta_{gc'}} (observed difference in Sanity's estimates)
\item \eqn{\eta_g^2 = \epsilon_{gc}^2 + \epsilon_{gc'}^2} (combined error variance)
\item \eqn{f_g(\alpha) = \alpha v_g/(\alpha v_g + \eta_g^2)} (shrinkage factor)
}

The shrinkage factor balances the observed gene expression differences
\eqn{x_g} against their measurement uncertainty \eqn{\eta_g}. For genes with
high-confidence estimates (\eqn{\eta_g \rightarrow 0}), it preserves the
observed differences while for noisy genes (\eqn{\eta_g \gg 0}), it shrinks
the result towards the common expected biological variation inferred from the
data (\eqn{\alpha v_g}).

The function returns the square root of the expected squared distance
\deqn{\langle d \rangle = \sqrt{\sum_g \langle \Delta_g^2 \rangle}}
}

\subsection{Hyperparameter \eqn{\alpha}}{

The key hyperparameter \eqn{\alpha} controls the prior distribution of \eqn{\Delta_g}:

\deqn{\Delta_g \sim N(0, \alpha v_g)}

Thus:
\itemize{
\item \eqn{\alpha = 0}: the 2 cells have identical expression states.
\item \eqn{\alpha = 2}: the 2 cells have independent expression states.
}

The function implements numerical integration over \eqn{\alpha} using a grid
of \code{nbin} values to compute the expected value of the squared distance across
all possible \eqn{\alpha}.
}

\subsection{Single to Noise Ratio (SNR)}{

\emph{Signal-to-Noise Ratio} (SNR) is defined as the ratio of the variance of
log-normalized counts across cells versus the mean variance (i.e. error bars)
for each genes.
}
}
\examples{
sce <- simulate_branched_random_walk(N_gene = 500, N_path = 10, length_path = 10)
sce <- Sanity(sce)  # necessary step before computing distances
d <- calculateSanityDistance(sce)

# Downstream analysis and visualization
hc <- hclust(d, method = "ward.D2")
plot(hc)

}
