% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spectra.R
\name{compareSpectra}
\alias{compareSpectra}
\alias{compareSpectra,Spectra,Spectra-method}
\alias{compareSpectra,Spectra,missing-method}
\title{Spectra similarity calculations}
\usage{
\S4method{compareSpectra}{Spectra,Spectra}(
  x,
  y,
  MAPFUN = joinPeaks,
  tolerance = 0,
  ppm = 20,
  FUN = ndotproduct,
  ...,
  matchedPeaksCount = FALSE,
  SIMPLIFY = TRUE
)

\S4method{compareSpectra}{Spectra,missing}(
  x,
  y = NULL,
  MAPFUN = joinPeaks,
  tolerance = 0,
  ppm = 20,
  FUN = ndotproduct,
  ...,
  matchedPeaksCount = FALSE,
  SIMPLIFY = TRUE
)
}
\arguments{
\item{x}{A \code{Spectra} object.}

\item{y}{A \code{Spectra} object.}

\item{MAPFUN}{For \code{compareSpectra()}: function to map/match peaks between
the two compared spectra. See \code{\link[=joinPeaks]{joinPeaks()}} for more information and
possible functions. Defaults to \code{\link[=joinPeaks]{joinPeaks()}}.}

\item{tolerance}{\code{numeric(1)} allowing to define a constant maximal
accepted difference between m/z values for peaks to be matched. This
parameter is directly passed to \code{MAPFUN}.}

\item{ppm}{\code{numeric(1)} defining a relative, m/z-dependent, maximal
accepted difference between m/z values for peaks to be matched. This
parameter is directly passed to \code{MAPFUN}.}

\item{FUN}{function to compare intensities of peaks between two spectra.
Defaults to \code{\link[MsCoreUtils:distance]{MsCoreUtils::ndotproduct()}}.}

\item{...}{Additional arguments passed to the internal functions.}

\item{matchedPeaksCount}{\code{logical(1)} whether the number of matching peaks
between the compared spectra should be returned in addition to the
similarity scores. Note that with \code{matchedPeaksCount = TRUE} a
3-dimensional \code{array} is returned. See examples below for details.}

\item{SIMPLIFY}{\code{logical(1)} defining  whether the result matrix should be
\emph{simplified} to a \code{numeric} if possible (i.e. if either \code{x} or \code{y} is
of length 1).}
}
\value{
For \code{matchedPeaksCount = FALSE} (the default) a \code{matrix} with the
similarity scores. With \code{matchedPeaksCount = FALSE} and
\code{SIMPLIFY = TRUE} a \code{numeric} vector. For \code{matchedPeaksCount = TRUE}
a 3-dimensional array with the scores reported in the first matrix in
z dimension (\verb{[, , 1]}) and the number of matching peaks in the second
matrix in z dimension (\verb{[, , 2]}).
}
\description{
\code{compareSpectra()} compares each spectrum in \code{x} with each spectrum in \code{y}
using the function provided with \code{FUN} (defaults to
\code{\link[MsCoreUtils:distance]{MsCoreUtils::ndotproduct()}}). If \code{y} is missing, each spectrum in \code{x} is
compared with each other spectrum in \code{x}.
The matching/mapping of peaks between the compared spectra is done with the
\code{MAPFUN} function. The default \code{\link[=joinPeaks]{joinPeaks()}} matches peaks of both spectra
and allows to keep all peaks from the first spectrum (\code{type = "left"}),
from the second (\code{type = "right"}), from both (\code{type = "outer"}) and to
keep only matching peaks (\code{type = "inner"}); see \code{\link[=joinPeaks]{joinPeaks()}} for more
information and examples). The \code{MAPFUN} function should have parameters
\code{x}, \code{y}, \code{xPrecursorMz} and \code{yPrecursorMz} as these values are passed to
the function.

In addition to \code{joinPeaks()} also \code{\link[=joinPeaksGnps]{joinPeaksGnps()}} is supported for
GNPS-like similarity score calculations. Note that \code{joinPeaksGnps()} should
only be used in combination with \code{FUN = MsCoreUtils::gnps}
(see \code{\link[=joinPeaksGnps]{joinPeaksGnps()}} for more information and details). Use
\code{MAPFUN = joinPeaksNone} to disable internal peak matching/mapping if a
similarity scoring function is used that performs the matching internally.

\code{FUN} is supposed to be a function to compare intensities of (matched)
peaks of the two spectra that are compared. The function needs to take two
matrices with columns \code{"mz"} and \code{"intensity"} as input and is supposed
to return a single numeric as result. In addition to the two peak matrices
the spectra's precursor m/z values are passed to the function as parameters
\code{xPrecursorMz} (precursor m/z of the \code{x} peak matrix) and \code{yPrecursorMz}
(precursor m/z of the \code{y} peak matrix). Additional parameters to functions
\code{FUN} and \code{MAPFUN} can be passed with \code{...}. Parameters \code{ppm} and
\code{tolerance} are passed to both \code{MAPFUN} and \code{FUN}.
The function returns a \code{matrix} with the results of \code{FUN} for each
comparison, number of rows equal to \code{length(x)} and number of columns
equal \code{length(y)} (i.e. element in row 2 and column 3 is the result from
the comparison of \code{x[2]} with \code{y[3]}). If \code{SIMPLIFY = TRUE} the \code{matrix}
is \emph{simplified} to a \code{numeric} if length of \code{x} or \code{y} is one. See also
the vignette for additional examples, such as using spectral entropy
similarity in the scoring.
}
\examples{

## Load a `Spectra` object with LC-MS/MS data.
fl <- system.file("TripleTOF-SWATH", "PestMix1_DDA.mzML",
    package = "msdata")
sps_dda <- Spectra(fl)
sps_dda

## Restrict to MS2 (fragment) spectra:
sps_ms2 <- filterMsLevel(sps_dda, msLevel = 2L)

## Compare spectra: comparing spectra 2 and 3 against spectra 10:20 using
## the normalized dotproduct method.
res <- compareSpectra(sps_ms2[2:3], sps_ms2[10:20])
## first row contains comparisons of spectrum 2 with spectra 10 to 20 and
## the second row comparisons of spectrum 3 with spectra 10 to 20
res

## Setting parameter `matchedPeaksCount = TRUE` returns in addition to the
## simialrity score also the number of matching peaks between the compared
## spectra. The results are then returned as a 3-dimensional array, with the
## first matrix in z dimension (`[, , 1]`) containing the scores and the
## second matrix in z dimention (`[, , 2]`) the number of matching peaks:
res <- compareSpectra(sps_ms2[2:3], sps_ms2[10:20], matchedPeaksCount = TRUE)

## the scores
res[, , 1L]

## the number of matching peaks
res[, , 2L]

## We next calculate the pairwise similarity for the first 10 spectra
compareSpectra(sps_ms2[1:10])

## Use compareSpectra to determine the number of common (matching) peaks
## with a ppm of 10:
## type = "inner" uses a *inner join* to match peaks, i.e. keeps only
## peaks that can be mapped betwen both spectra. The provided FUN returns
## simply the number of matching peaks.
compareSpectra(sps_ms2[2:3], sps_ms2[10:20], ppm = 10, type = "inner",
    FUN = function(x, y, ...) nrow(x))

## We repeat this calculation between all pairwise combinations
## of the first 20 spectra
compareSpectra(sps_ms2[1:20], ppm = 10, type = "inner",
    FUN = function(x, y, ...) nrow(x))

}
\author{
Sebastian Gibb, Johannes Rainer, Laurent Gatto
}
