% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anglemania_utils.R
\name{anglemania_utils}
\alias{anglemania_utils}
\alias{sparse_to_fbm}
\alias{replace_with_na}
\alias{normalize_matrix}
\alias{get_anglemania_genes}
\alias{get_anglemania_stats_df}
\title{Utility Functions for the anglemania Package}
\usage{
sparse_to_fbm(s_mat)

replace_with_na(v)

normalize_matrix(
  x_mat,
  normalization_method = "divide_by_total_counts",
  verbose = TRUE
)

get_anglemania_genes(sce)

get_anglemania_stats_df(sce)
}
\arguments{
\item{s_mat}{A sparse matrix.}

\item{v}{A numeric vector.}

\item{x_mat}{A \code{bigstatsr::FBM} object containing the matrix to normalize
(typically genes x cells).}

\item{normalization_method}{A character string specifying the normalization
method to use. One of \code{"divide_by_total_counts"} (default) or
\code{"find_residuals"}.
\itemize{
\item \code{"divide_by_total_counts"} normalizes each cell by its total expression
count and applies log1p.
\item \code{"find_residuals"} computes log1p-transformed residuals after regressing
out total expression.
}}

\item{sce}{A SingleCellExperiment or SummarizedExperiment object}
}
\value{
An \code{\link[bigstatsr]{FBM}} object from the \pkg{bigstatsr}
package.

A numeric vector with NaN and Inf values replaced with NA.

The input \code{FBM} object with normalized values written back in place.
This function modifies the input \code{x_mat} by reference.

A character vector of gene names that have been selected by the
anglemania algorithm

A data frame of gene pairs from which the anglemania genes
were selected
}
\description{
A collection of utility functions used within the
\pkg{anglemania} package
for manipulating FBMs, calculating statistics, and selecting genes.

Replaces all NaN and Inf values in a numeric vector with NA.
}
\section{Functions}{
\itemize{
\item \code{sparse_to_fbm()}: Convert a sparse matrix into
a file-backed matrix
(\code{\link[bigstatsr]{FBM}}) with efficient memory usage.

\item \code{replace_with_na()}: replace Nan and Inf values with NA

\item \code{normalize_matrix()}: normalize matrix
Normalize a Filebacked Big Matrix (\code{FBM}) using either total-count
scaling or residuals from a linear model. Intended for use with
single-cell RNA-seq gene expression data.

\item \code{get_anglemania_genes()}: Utility function to extract the genes that
have been selected by the anglemania algorithm.

\item \code{get_anglemania_stats_df()}: Utility function to extract the stats of the
gene pairs from which the anglemania genes were selected.

}}
\examples{
s_mat <- Matrix::rsparsematrix(nrow = 10, ncol = 5, density = 0.3)
fbm_mat <- sparse_to_fbm(s_mat)
fbm_mat
v <- c(1, 2, 3, 4, 5, 6, 7, Inf, 9, NA)
v <- replace_with_na(v)
v
library(bigstatsr)
set.seed(42)
mat <- matrix(rpois(1000, lambda = 5), nrow = 100, ncol = 10)
fbm <- as_FBM(mat)

normalize_matrix(
  fbm,
  normalization_method = "divide_by_total_counts"
)[1:5, 1:5]
normalize_matrix(
  fbm,
  normalization_method = "find_residuals"
)[1:5, 1:5]

sce <- sce_example()
sce <- anglemania(sce, batch_key = "batch")
anglemania_genes <- get_anglemania_genes(sce)
head(anglemania_genes)
length(anglemania_genes)
sce <- sce_example()
sce <- anglemania(sce, batch_key = "batch")
anglemania_stats_df <- get_anglemania_stats_df(sce)
head(anglemania_stats_df)
length(anglemania_stats_df)
}
\keyword{internal}
