% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_genes.R
\name{select_genes}
\alias{select_genes}
\alias{prefilter_angl}
\alias{extract_rows_for_unique_genes}
\title{Select genes from an anglemania-processed SCE}
\usage{
prefilter_angl(
  sce,
  zscore_mean_threshold = 1,
  zscore_sn_threshold = 1,
  verbose = TRUE
)

select_genes(
  sce,
  max_n_genes = 2000,
  score_weights = c(0.4, 0.6),
  verbose = TRUE
)

extract_rows_for_unique_genes(dt, max_n_genes)
}
\arguments{
\item{sce}{A \code{SingleCellExperiment} object.}

\item{zscore_mean_threshold}{Numeric value specifying the threshold for the
absolute mean z-score. Default is 1.}

\item{zscore_sn_threshold}{Numeric value specifying the threshold for the
SNR z-score. Default is 1.}

\item{verbose}{Logical value indicating whether to print progress messages.
Default is \code{TRUE}.}

\item{max_n_genes}{An integer specifying the maximum number of unique genes
to return.}

\item{score_weights}{A vector of two numeric values specifying the weights
for the mean z-score and standard deviation of z-score, respectively.
Default is \code{c(0.4, 0.6)} for a greater emphasis on the standard
deviation of z-score.}

\item{dt}{A data frame containing gene pairs, with columns \code{geneA}
and \code{geneB}.}
}
\value{
A data frame containing the prefiltered gene pairs.

The input \code{SingleCellExperiment} object with the
\code{anglemania_genes} slot updated to include the selected genes and
their statistical information.

A vector of unique gene identifiers.
}
\description{
Select genes from a SingleCellExperiment object based on
mean z-score and the signal-to-noise ratio of angles between gene pairs
across batches.
}
\details{
The function performs the following steps:
\enumerate{
\item Identifies gene pairs where both the mean z-score and SNR z-score
exceed the specified thresholds.
}

Selects the top n genes based on the weighted sum of the ranked mean
and standard deviation of the z-score of the correlations between gene pairs.

The function combines the \code{geneA} and \code{geneB} columns, extracts
unique gene names, and returns the first \code{max_n_genes} genes. If
\code{max_n_genes} exceeds the number of unique genes available, all unique
genes are returned.
}
\section{Functions}{
\itemize{
\item \code{prefilter_angl()}: Prefilter gene pairs from the mean and SNR z-scores
based on thresholds, to simplify downstream filtering.

\item \code{select_genes()}: Select the top n genes on the weighted sum
of the ranks of the mean z-score and SNR z-score of the gene pairs.

\item \code{extract_rows_for_unique_genes()}: Extract unique gene identifiers
from gene pairs, returning up to a specified maximum number.

}}
\examples{
library(SingleCellExperiment)
sce <- sce_example()
sce <- anglemania(sce, batch_key = "batch")
prefiltered_df <- prefilter_angl(
  sce,
  zscore_mean_threshold = 1,
  zscore_sn_threshold = 1
)
head(prefiltered_df)
sce <- sce_example()
sce <- anglemania(
    sce,
    batch_key = "batch",
    max_n_genes = 20
)
anglemania_genes <- get_anglemania_genes(sce)
# View the selected genes and use for integration
head(anglemania_genes)
length(anglemania_genes)
sce <- select_genes(
    sce,
    max_n_genes = 10
)
anglemania_genes <- get_anglemania_genes(sce)
head(anglemania_genes)
length(anglemania_genes)
gene_pairs <- data.frame(
  geneA = c("Gene1", "Gene2", "Gene3", "Gene4"),
  geneB = c("Gene3", "Gene4", "Gene5", "Gene6")
)
unique_genes <- extract_rows_for_unique_genes(
  gene_pairs,
  max_n_genes = 3
)
print(unique_genes)
}
\seealso{
\code{\link{extract_rows_for_unique_genes}},
\code{\link{get_intersect_genes}}, \code{\link{get_list_stats}}

\code{\link{select_genes}}
}
\keyword{internal}
