#' Differential proportion test on Barcodes across sample groups
#'
#' @param proportion The `proportion` assay in `SummarizedExperiment::assays(barbieQ)`.
#' @param count The main assay of `barbieQ`, indicating raw counts of barcodes.
#' @param transformation A string specifying the transformation method.
#'  Options include: 'asin-sqrt', 'logit', and 'none'.
#'  Defaults to 'asin-sqrt'.
#' @param mycontrasts A numeric vector generated by [limma::makeContrasts]
#' @param designMatrix A numeric matrix standardizing `targets`, generated by
#'  the [stats::model.matrix] function. Defaults to be generated by
#'  `designFormula`.
#' @param block A vector (array) indicating sample duplicates. Defaults to
#'  no duplicates among the samples.
#'
#' @importFrom limma duplicateCorrelation
#' @importFrom limma lmFit
#' @importFrom limma contrasts.fit
#' @importFrom limma eBayes
#' @importFrom limma decideTests
#' @importFrom magrittr %>%
#' @importFrom SummarizedExperiment assays
#' @importClassesFrom SummarizedExperiment SummarizedExperiment
#'
#' @return A `data.frame` of statistical test results for each Barcode
#'
#' @noRd
#'
#' @examples \donttest{
#' Block <- c(1, 1, 2, 3, 3, 4, 1, 1, 2, 3, 3, 4)
#' Treat <- factor(rep(c('ctrl', 'drug'), each = 6))
#' Time <- rep(rep(seq_len(2), each = 3), 2)
#' nbarcodes <- 50
#' nsamples <- 12
#' count <- abs(matrix(rnorm(nbarcodes * nsamples), nbarcodes, nsamples))
#' rownames(count) <- paste0('Barcode', seq_len(nbarcodes))
#' barbieQ <- createBarbieQ(count, data.frame(Treat = Treat, Time = Time))
#' barbieQ:::testDiffProp(
#'   proportion = SummarizedExperiment::assays(barbieQ)$proportion,
#'   mycontrasts = c(-1, 1, 0),
#'   designMatrix = model.matrix(~ 0 + Treat + Time)
#' )
#' }
testDiffProp <- function(proportion, count, transformation = "asin-sqrt", mycontrasts = NULL, designMatrix = NULL,
    block = NULL) {
    ## checking transformation
    transformation <- match.arg(transformation, c("asin-sqrt", "logit", "none"))
    ## proportion assay from`barbieQ`

    ## apply the appropriate transformation
    if (transformation == "asin-sqrt") {
        mydata <- asin(sqrt(proportion))
    } else if (transformation == "logit") {
        countPlus <- count + 0.5
        proportionPlus <- countPlus / colSums(countPlus)
        mydata <- log((proportionPlus)/(1 - proportionPlus))
    } else {
        mydata <- proportion
    }

    if (!is.null(block)) {
        ## compute duplicates correction
        dup <- limma::duplicateCorrelation(object = mydata, design = designMatrix, block = block)
        message("Consensus correlation of Barcode proportion within sample duplicates: ",
            dup$consensus.correlation)
        ## fit limma linear regression model taking duplicates
        myfit1 <- limma::lmFit(object = mydata, design = designMatrix, block = block, correlation = dup$consensus.correlation)
    } else {
        ## ignore duplicates
        message("no block specified, so there are no duplicate measurements.")
        ## fit limma linear regression model without duplicates
        myfit1 <- limma::lmFit(object = mydata, design = designMatrix)
    }
    ## fit contrast
    myfit2 <- limma::contrasts.fit(fit = myfit1, contrasts = mycontrasts)
    ## fit eBayes, moderated p.values obtained after applying empirical Bayes
    myfit3 <- limma::eBayes(myfit2)
    ## extract test result using multiple testing adjusted p.values
    myresults <- limma::decideTests(myfit3, adjust.method = "BH")

    ## save stats to a data.fame
    statsDf <- data.frame(meanDiff = myfit3$coefficients, Amean = myfit3$Amean, t = myfit3$t,
        P.Value = myfit3$p.value, adj.P.Val = p.adjust(myfit3$p.value, method = "BH"), direction = myresults)

    return(statsDf)
}
