% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/templates.R, R/TemplatesDoc.R
\name{TemplatesFunctions}
\alias{TemplatesFunctions}
\alias{update_template_cvg}
\alias{adjust_binding_regions}
\alias{assign_binding_regions}
\alias{select_regions_by_conservation}
\title{Template Functionalities.}
\usage{
update_template_cvg(template.df, primer.df, mode.directionality = NULL)

adjust_binding_regions(template.df, region.fw, region.rev)

assign_binding_regions(
  template.df,
  fw = NULL,
  rev = NULL,
  optimize.region = FALSE,
  primer.length = 20,
  gap.char = "-"
)

select_regions_by_conservation(
  template.df,
  gap.char = "-",
  win.len = 30,
  by.group = TRUE,
  mode.directionality = c("both", "fw", "rev")
)
}
\arguments{
\item{template.df}{An object of class \code{Templates}.}

\item{primer.df}{An object of class \code{Primers} containing
primers with annotated coverage that are to be used to update 
the template coverage in \code{template.df}.}

\item{mode.directionality}{The directionality of primers/templates.}

\item{region.fw}{Interval of new binding regions relative to the forward binding region defined in \code{template.df}.}

\item{region.rev}{Interval of new binding regions relative to the reverse binding region defined in \code{template.df}}

\item{fw}{Binding regions for forward primers. Either a numeric interval indicating a uniform
binding range relative to the template 5' end or a path to a FASTA file providing
binding sequences for every template. If \code{fw} is missing, only
\code{rev} is considered.}

\item{rev}{Binding regions for reverse primers. Either a numeric interval indicating a uniform
binding range relative to the template 3' end or the path to a FASTA file providing
binding sequences for every template. If \code{rev} is missing,
only \code{fw} is considered.}

\item{optimize.region}{If \code{TRUE}, the binding regions
specified via \code{fw} and \code{rev} are 
adjusted such that binding regions that may form secondary structures are 
avoided. This feature requires ViennaRNA (see notes). If \code{FALSE}
(the default), the input binding regions are not modified.}

\item{primer.length}{A numeric scalar providing the probe length that is used for
adjusting the primer binding regions when \code{optimize.region} is \code{TRUE}.}

\item{gap.char}{The character in the input file representing gaps.}

\item{win.len}{The extent of the desired primer binding region.
This should be smaller than the \code{allowed.region}. The default is 30.}

\item{by.group}{Shall the determination of binding regions be stratified
according to the groups defined in \code{template.df}. By default,
this is set to \code{TRUE}.}
}
\value{
\code{update_template_cvg} returns an object of class 
\code{Templates} with updated coverage columns.

\code{adjust_binding_regions} returns a \code{Templates} object with
updated binding regions.

\code{assign_binding_regions} returns an object of class \code{Templates} with newly assigned binding regions.

\code{select_regions_by_conservation} returns a
\code{Templates} object with adjusted binding regions.
The attribute \code{entropies} gives a data frame with positional entropies
and the attribute \code{alignments} gives the alignments of the templates.
}
\description{
\describe{
\item{\code{adjust_binding_regions}}{Adjusts the existing annotation 
of binding regions by specifying
a new binding interval relative to the existing binding region.}
\item{\code{assign_binding_regions}}{Assigns the primer target binding 
regions to a set of template sequences.}
\item{\code{update_template_cvg}}{Annotates the template coverage.}
\item{\code{select_regions_by_conservation}}{Computes Shannon entropy
for the defined binding regions and determines the most conserved regions.}
}
}
\details{
When modifying binding regions with \code{adjust_binding_regions}, new
binding intervals can be specified via \code{fw} and \code{rev}
for forward and reverse primers, respectively. The new regions should
be provided relative to the existing definition of binding regions 
in \code{template.df}.
For specifying the new binding regions, position \code{0} refers to 
the first position after the end of the existing
binding region. Hence, negative positions relate to regions within the
existing binding region, while non-negative values relate to positions
outside the defined binding region.

Binding regions are defined using \code{assign_binding_regions}, where
the arguments \code{fw} and \code{rev} provide data describing
the binding regions of the forward and reverse primers, respectively.
To specify binding regions for each template
individually, \code{fw} and \code{rev} should provide the paths to FASTA files. The headers of these
FASTA file should match the headers of the loaded \code{template.df} 
and the sequences in the files specified by \code{fw} and \code{rev} should indicate
the target binding regions. 

To specify uniform binding regions,
\code{fw} and \code{rev} should be numeric intervals indicating the allowed
binding range for primers in the templates. Setting the forward interval
to (1,30) indicates that the first 30 bases should be used for forward primers
and specifying the reverse interval to (1,30) indicates that the last
30 bases should be used for reverse primer binding.

If \code{optimize.region} is \code{TRUE}, the input binding region is
adjusted such that regions forming secondary structures are avoided.
}
\note{
\code{assign_binding_regions} requires the program ViennaRNA
(https://www.tbi.univie.ac.at/RNA/)
for adjusting the binding regions when \code{optimize.region} 
is set to \code{TRUE}.

\code{select_regions_by_conservation} requires the MAFFT software
for multiple alignments 
(http://mafft.cbrc.jp/alignment/software/).
}
\examples{

# Annotate the coverage of the templates
data(Ippolito)
template.df <- update_template_cvg(template.df, primer.df)
data(Ippolito)
# Extend the binding region by one position
relative.interval <- c(-max(template.df$Allowed_End_fw), 0)
template.df.adj <- adjust_binding_regions(template.df, relative.interval)
# compare old and new annotations:
head(cbind(template.df$Allowed_Start_fw, template.df$Allowed_End_fw))
head(cbind(template.df.adj$Allowed_Start_fw, template.df.adj$Allowed_End_fw))
data(Ippolito)
# Assignment of individual binding regions
l.fasta.file <- system.file("extdata", "IMGT_data", "templates", 
     "Homo_sapiens_IGH_functional_leader.fasta", package = "openPrimeR")
template.df.individual <- assign_binding_regions(template.df, l.fasta.file, NULL)
# Assign the first/last 30 bases as forward/reverse binding regions
template.df.uniform <- assign_binding_regions(template.df, c(1,30), c(1,30))
# Optimization of binding regions (requires ViennaRNA)
\dontrun{template.df.opti <- assign_binding_regions(template.df, c(1,30), c(1,30),
                     optimize.region = TRUE, primer.length = 20)}
data(Ippolito)
new.template.df <- select_regions_by_conservation(template.df)
}
