#' @title Plot networks created by GatherNetworks function
#' @name plotNetworks
#' @description Helper function for plotting networks of metabolites gathered
#' from the KEGG pathways database using the 
#' \code{\link[pairkat]{GatherNetworks}} function.
#'
#' @param networks networks object obtained
#' with \code{\link[pairkat]{GatherNetworks}}
#' @param pathway Pathway to be plotted. Leaving this as
#' 'all' will plot all pathways in `networks`
#' @param ... Parameters to be passed to the \code{\link[base]{plot}}
#' (i.e. \code{\link[igraph]{plot.igraph}}) function
#'
#' @details Plots the specified network(s) as an \code{\link[igraph]{igraph}}
#'
#' @return
#' a plot or list of plots generated by igraph
#'
#' @examples
#' data(smokers)
#'
#' # Query KEGGREST API
#' networks <- GatherNetworks(SE = smokers, keggID = "kegg_id",
#' species = "hsa", minPathwaySize = 5)
#'
#' # Plot all networks
#' plotNetworks(networks)
#'
#' # Plot specific network
#' plotNetworks(networks,
#' pathway = "Glycerophospholipid metabolism",
#' layout = igraph::layout_with_kk,
#' main = "Glycerophospholipid Metabolism")
#' 
#' @importFrom igraph plot.igraph
#' @importFrom igraph layout_with_kk
#'
#' @export
plotNetworks <- function(networks, pathway = 'all', ...) {
    if (pathway == 'all')
        lapply(networks$networks, plot, ...)
    
    else
        plot(networks$networks[[pathway]], ...)
}
