#' Pie plot from PAC
#'
#' \code{PAC_pie} Pie chart of annotation features (anno_target).
#'
#' Given a PAC object the function will summarize the counts into percent
#' of a specified anno_target and plot a pie chart.
#'
#' @family PAC analysis
#'
#' @seealso \url{https://github.com/OestLab/seqpac} for updates on the current
#'   package.
#'
#' @param PAC PAC-list object.
#' 
#' @param anno_target List with: 
#'                          1st object being character vector of target
#'                          column(s) in Anno, 2nd object being a character
#'                          vector of the target biotype(s) in the target column
#'                          (1st object). Important, the 2nd object is order
#'                          sensitive, meaning that categories will appear in
#'                          the same order in the pie. (default=NULL)
#'
#'
#' @param pheno_target List with: 
#'                          1st object being character vector of target
#'                          column(s) in Pheno, 2nd object being a character
#'                          vector of the target group(s) in the target column
#'                          (1st object). Important, the 2nd object is order
#'                          sensitive, meaning that categories will appear in
#'                          the same order in the pie. (default=NULL)
#'                          
#' @param summary_target List with 1st object being a character of which summary
#' to use, produced by \code{PAC_summary}. (default=NULL) 
#' 
#' @param norm Character vector defining what normalized data to plot. 
#' (default="counts")
#'
#' @param colors Character vector RGB color codes as generated by for example
#'   grDevices::colorRampPalette. If colors=NULL (default), colors will be
#'   generated using a default color palette. Important: In default mode,
#'   categories named "other" or no_anno" will automatically receive a grey
#'   color.
#'
#' @param angle Integer (positive or negative) that sets the rotation of
#'   the pie.
#'   
#' @param labels Character that sets what labels to plot in the actual pie
#'   besides the legend. If lables="all" (default), then the anno_target variables will be
#'   combined with percentage. If lables="percent" only percentages will be
#'   plotted with the pie. If labels="none", no lables will be present
#'   besides the legend.
#'   
#' @param no_anno Logical whether PAC sequences without an annotation should be
#'   removed prior to plotting. Specifically, if no_anno=FALSE, then sequences
#'   annotated with "no_anno" in the anno_target column will be removed prior to
#'   plotting. When no_anno=TRUE (default), then all sequences will be included
#'   (unless excluded in the anno_target object).
#'   
#' @return A pie plot
#'
#' @examples
#' 
#' 
#' ##########################################
#' ### Pie charts in seqpac 
#' ##----------------------------------------
#' 
#' load(system.file("extdata", "drosophila_sRNA_pac_filt_anno.Rdata", 
#'                   package = "seqpac", mustWork = TRUE))
#' 
#' # Choose an anno_target and plot all samples (deafult):
#' output_pie <- PAC_pie(pac, anno_target=list("Biotypes_mis0"))
#' output_pie[[1]]
#' output_pie[[6]]
#' 
#' # Defining a summary_target will show summarised data:
#' pac<-PAC_summary(pac)
#' PAC_pie(pac, anno_target=list("Biotypes_mis0"), 
#' summary_target=list("countsMeans_All"))
#' # Rotate:
#' PAC_pie(pac, anno_target=list("Biotypes_mis0"), 
#' summary_target=list("countsMeans_All"), angle=180)
#' 
#' 
#' #  Make ordered pie charts of grand mean percent of all samples
#' ord_bio <- as.character(sort(unique(anno(pac)$Biotypes_mis3)), 
#'                                     unique(anno(pac)$Biotypes_mis0))
#' output_pie_1 <- PAC_pie(pac, anno_target=list("Biotypes_mis0", ord_bio), 
#'                         labels = "percent", summary_target=list("countsMeans_All"))
#' output_pie_2 <- PAC_pie(pac, anno_target=list("Biotypes_mis3", ord_bio), 
#'                         labels = "percent", summary_target=list("countsMeans_All"))
#' cowplot::plot_grid(plotlist=c(output_pie_1, output_pie_2), nrow=2, 
#'                    scale = 1.0)
#' 
#' #  Shortcut to remove no annotations ("no_anno") in the anno_target column
#' PAC_pie(pac, anno_target=list("Biotypes_mis3"), summary_target=list("countsMeans_All"),
#'  no_anno=TRUE)
#' PAC_pie(pac, anno_target=list("Biotypes_mis3"), summary_target=list("countsMeans_All"), 
#' no_anno=FALSE)
#' 
#' @export


PAC_pie <- function(PAC, anno_target=NULL, pheno_target=NULL, colors=NULL, norm="counts",
                    labels="all", no_anno=TRUE, summary_target=NULL, angle=-25){
  
  ## Check S4
  if(isS4(PAC)){
    tp <- "S4"
    PAC <- as(PAC, "list")
  }else{
    tp <- "S3"
  }
  
  quiet <- function(x) { 
    sink(tempfile()) 
    on.exit(sink()) 
    invisible(force(x)) 
    }
  
  stopifnot(PAC_check(PAC))
  types <- variables <- NULL
  
  ## Prepare targets
  if(!is.null(pheno_target)){ 
    if(length(pheno_target)==1){ 
      pheno_target[[2]] <- as.character(unique(PAC$Pheno[,pheno_target[[1]]]))
    }
  }else{
    PAC$Pheno$eXtra_Col <- rownames(PAC$Pheno)
    pheno_target <- list(NA)
    pheno_target[[1]] <-  "eXtra_Col"
    pheno_target[[2]] <-  PAC$Pheno$eXtra_Col
    }
  if(!is.null(anno_target)){ 
    if(length(anno_target)==1){ 
      anno_target[[2]] <- as.character(unique(PAC$Anno[,anno_target[[1]]]))
    }
  }

  ## Subset
  PAC_sub <- PAC_filter(PAC, subset_only=TRUE, pheno_target=pheno_target, 
                        anno_target=anno_target)
  ### Removing no_Anno
  if(no_anno==FALSE){
    PAC_sub <- PAC_filter(PAC_sub, anno_target=list(anno_target[[1]], "no_anno"), 
                         allbut=TRUE)
  }
  
  anno <- PAC_sub$Anno
  pheno <- PAC_sub$Pheno
  
  if(norm == "counts"){
    data <- PAC_sub$Counts
  }
  else{
    data <- PAC_sub$norm[norm][[1]]
  }
  
  if(!is.null(summary_target)){
    data <- PAC_sub$summary[summary_target[[1]]]
    data <- data[[1]]
  }

  tot_cnts <- colSums(data)
  data_shrt <- stats::aggregate(data, list(anno[, anno_target[[1]]]), "sum")
  
  data_shrt_perc <- data_shrt
  data_shrt_perc[,-1] <- "NA"
  for (i in seq.int(length(tot_cnts))){ 
    data_shrt_perc[,1+i]   <- data_shrt[,1+i]/tot_cnts[i]
  }
  
  data_long_perc <- reshape2::melt(data_shrt_perc, id.vars="Group.1")
  colnames(data_long_perc) <- c("Category", "Sample", "Percent")
  data_long_perc$Percent <- data_long_perc$Percent*100

  ## Fix order
  # Anno
  bio <- anno_target[[2]] 
  extra  <- which(bio %in% c("no_anno", "other"))
  bio <- bio[c(extra, (seq.int(length(bio)))[!seq.int(length(bio)) %in% extra])] 
  data_long_perc$Category <- factor(as.character(data_long_perc$Category), 
                                    levels=bio)

  ### Plot data
  if(is.null(colors)){
    n_extra  <- length(extra)
    colfunc <- grDevices::colorRampPalette(c("#094A6B", "#EBEBA6", "#9D0014"))
    if(n_extra==1){colors <- c(colfunc(length(bio)-1), "#6E6E6E")}
    if(n_extra==2){colors <- c(colfunc(length(bio)-2), "#6E6E6E", "#BCBCBD")}
    if(n_extra==0){colors <- colfunc(length(bio))}
  }
  
  prec_lst <- split(data_long_perc, data_long_perc$Sample)   
  prec_lst <- lapply(prec_lst, function(x){
    x <- x[match(levels(x$Category), x$Category),,drop=FALSE]
    x$Category <- factor(levels(x$Category), levels=levels(x$Category))
    x$Percent[is.na(x$Percent)] <- 0
    return(x)
  })
  plt_lst<- quiet(lapply(prec_lst, function(x){
    # setup labels
    if(labels=="all"){
      labs <- paste0(x$Category, "_", round(x$Percent, digits=0), "%")
    }
    if(labels=="percent"){  
      labs <- paste0(round(x$Percent, digits=0), "%")
    }
    if(labels=="none"){ 
      labs <- NA 
    }
    # Position of labels and plot
    x <- dplyr::arrange(x, dplyr::desc(Category))
    x$prop <- x$Percent / sum(x$Percent) * 100
    x$ypos <- cumsum(x$prop) - 0.5 * x$prop
    
    p1  <-ggplot2::ggplot(x, ggplot2::aes(x="", y=prop, fill=Category)) +
      ggplot2::geom_bar(stat="identity", width=1, color="white") +
      ggplot2::coord_polar("y", start=angle) +
      ggplot2::theme_void() + 
      ggplot2::theme(legend.position="none") +
      ggplot2::geom_segment(ggplot2::aes(x = 1.5, xend = 1.56,
                                         y = ypos, yend = ypos),
                            color = "black", linewidth = 0.5) +
      
      ggplot2::geom_text(ggplot2::aes(x= 1.7,y = ypos, label = rev(labs)), color = "black", size=4) +
      ggplot2::scale_fill_manual(values=rev(colors))
    print(p1)
    }))
  graphics::plot.new()
  df <- data.frame(types=prec_lst[[1]]$Category, 
                   variables=prec_lst[[1]]$Category, 
                   levels=levels(prec_lst[[1]]$Category)) 
  suppressWarnings(leg <- cowplot::get_legend(ggplot2::ggplot(
    df, ggplot2::aes(x=types, fill=variables)) + 
                               ggplot2::geom_bar(color="black") + 
                               ggplot2::scale_fill_manual(values=rev(colors))))
  return(c(plt_lst, list(legend=leg)))
}
