calculate_average_neutron_mass <- function(element_array) {
    total_mass <- 0
    total_atoms <- 0
    for (i in seq_along(element_array)) {
        atom_count <- element_array[i]
        isotopes <- supported_isotopes[[names(element_array)[i]]]
        if (length(isotopes) > 1 && atom_count > 0) {
            for (j in 2:length(isotopes)) {
                isotope <- isotopes[j]
                delta_mass <- isotopic_masses[isotope] - isotopic_masses[isotopes[j - 1]]
                total_mass <- total_mass + delta_mass * atom_count *
                    shared_env$isotopic_abundances[isotope]
                total_atoms <- total_atoms + atom_count *
                    shared_env$isotopic_abundances[isotope]
            }
        }
    }
    average_neutron_mass <- total_mass / total_atoms
    return(average_neutron_mass)
}

isotope_abundance_fft <- function(isotope_abundance, N_width) {
    isotope_abundance <- c(isotope_abundance, rep(0, N_width - length(isotope_abundance)))
    isotope_abundance_fft <- fft(isotope_abundance)
    return(isotope_abundance_fft)
}

cal_isotope_abundance_fft <- function(element_array, N_width) {
    abundance_fft <- 1
    for (i in seq_along(element_array)) {
        atom_count <- element_array[i]
        isotopes <- supported_isotopes[[names(element_array)[i]]]
        if (length(isotopes) > 1 && atom_count > 0) {
            abundance_fft <- abundance_fft *
                isotope_abundance_fft(shared_env$isotopic_abundances[isotopes], N_width)**
                    atom_count
        }
    }
    abundance_fft <- fft(abundance_fft, inverse = TRUE)
    abundance_fft <- Re(abundance_fft)
    abundance_fft <- abundance_fft / sum(abundance_fft)
    return(abundance_fft)
}


#' Calculate Isotope Peaks using FFT
#'
#' This function calculates the isotope peaks for a given chemical formula using
#' Fast Fourier Transform (FFT).  It approximates the delta mass of isotopes is one neutron mass
#' and ignore the fine structure of isotopes.
#'
#' @param formula A character string representing the chemical formula.
#' @param N_width An integer specifying the width of the isotope envelope in FFT. Default is 100.
#' Wider isotopic envolope will require larger N_width.
#' @param min_abundance A numeric value specifying the minimum abundance threshold
#' for the peaks. Default is 0.0001.
#' @param ... Additional arguments passed to the function. For example C13=0.5 will change
#' the abundance of C13 to 0.5 and adjust the abundance of C12 accordingly.
#'
#' @return data.frame A data frame containing the calculated isotope peaks mass and their abundances.
#'
#' @examples
#' # Example usage:
#' cal_isotope_peaks_fft("C6H12O6")
#' cal_isotope_peaks_fft("C6H12O6", N_width = 200, min_abundance = 0.001, C13 = 0.5)
#'
#' @export
cal_isotope_peaks_fft <- function(formula, N_width = 100, min_abundance = 0.0001, ...) {
    # Save the original isotopic abundances
    original_abundances <- shared_env$isotopic_abundances

    # Parse the variable parameters
    params <- list(...)
    for (param in names(params)) {
        if (param %in% names(shared_env$isotopic_abundances)) {
            shared_env$isotopic_abundances[param] <- params[[param]]
            element <- sub("[0-9]+", "", param)
            other_isotopes <- setdiff(supported_isotopes[[element]], param)
            remaining_abundance <- 1 - params[[param]]
            for (other_isotope in other_isotopes) {
                shared_env$isotopic_abundances[other_isotope] <- shared_env$isotopic_abundances[other_isotope] / sum(shared_env$isotopic_abundances[other_isotopes]) * remaining_abundance
            }
        } else {
            warning(
                sprintf(
                    "Unsupported isotope: %s will be ignored.",
                    toString(param)
                ),
                call. = FALSE
            )
        }
    }

    element_array <- parse_chemical_formula(formula)
    average_neutron_mass <- calculate_average_neutron_mass(element_array)
    monoisotopic_mass <- cal_monoisotopic_mass(formula)
    isotope_abundances <- cal_isotope_abundance_fft(element_array, N_width)
    isotope_abundances <- isotope_abundances[isotope_abundances > min_abundance]
    isotope_masses <- monoisotopic_mass + average_neutron_mass * (0:(length(isotope_abundances) - 1))

    # Recover the original isotopic abundances
    shared_env$isotopic_abundances <- original_abundances

    return(data.frame(Mass = isotope_masses, Prob = isotope_abundances))
}

#' Plot Molecular isotopes without fine structure by FFT algorithm
#'
#' This function plots the molecular isotopes generated by Fast Fourier Transform (FFT).
#'
#' @param isotope_numbers A data.frame representing the isotope mass and abundance to be plotted.
#' @param charge An integer representing the charge. Default is 1.
#' @param minProb A numeric value representing the minimum probability. Default is 0.0001.
#' @param yshift A numeric value representing the vertical shift applied to the plot for better visualization of the abundance close to 0. Default is -1.
#' @param peakWidth A numeric value representing the width of the peaks in the plot. Default is 0.5.
#' @param textSize A numeric value representing the size of the text in the plot.
#'
#' @return A ggplot object of molecular isotopes without fine structure by FFT algorithm
#' @export
#'
#' @examples
#' isotope_numbers <- cal_isotope_peaks_fft("C6H12O6", N_width = 200, min_abundance = 0.001, C13 = 0.5)
#' plotMolecularFFTisotopes(isotope_numbers)
plotMolecularFFTisotopes <- function(isotope_numbers, charge = 1, minProb = 0.0001, yshift = -1, peakWidth = 0.5, textSize = 15) {
    isotope_numbers <- isotope_numbers[isotope_numbers$Prob > minProb, ]
    spectra <- data.frame(
        Mass = isotope_numbers$Mass,
        Prob = isotope_numbers$Prob
    )
    spectra$MZ <- spectra$Mass / charge
    spectra$Charge <- charge
    # for plot
    maxProb <- max(spectra$Prob)
    spectra$Prob <- spectra$Prob / maxProb * 100
    spectra <- dplyr::arrange(spectra, MZ)
    spectra$Prob <- spectra$Prob
    spectra$Kind <- paste0(spectra$Charge)
    p <- ggplot2::ggplot(spectra)
    p <- p + ggplot2::aes(x = MZ, ymax = Prob, ymin = yshift)
    p <- p + ggplot2::geom_linerange(linewidth = peakWidth)
    p <- p + ggplot2::scale_x_continuous(breaks = seq(min(spectra$MZ) - 1, max(spectra$MZ) + 1, by = 1))
    p <- p + ggplot2::scale_y_continuous(breaks = seq(0, 100, by = 10))
    p <- p + ggplot2::theme(
        panel.grid = ggplot2::element_blank(),
        panel.background = ggplot2::element_blank(),
        legend.key = ggplot2::element_blank(),
        panel.border = ggplot2::element_rect(
            fill = NA,
            color = "grey10",
            linetype = 1,
            linewidth = 0.5
        ),
        text = ggplot2::element_text(size = textSize)
    ) +
        ggplot2::xlab("M/Z") +
        ggplot2::ylab("Intensity") +
        ggplot2::guides(color = ggplot2::guide_legend(
            override.aes =
                list(
                    linewidth = 5, fill =
                        NA
                )
        ))
    return(p)
}
