## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment  = "#>",
    fig.path = "figures/",
    fig.crop = FALSE,
    warning  = FALSE,
    message  = FALSE
)

## ----installation, eval = FALSE-----------------------------------------------
# # Requires R >= 4.5.0
# if (!require("BiocManager", quietly = TRUE)) install.packages("BiocManager")
# BiocManager::install("BreastSubtypeR")

## ----citation, eval = FALSE---------------------------------------------------
# citation("BreastSubtypeR")

## ----pkgs-and-data, message=FALSE---------------------------------------------
library(BreastSubtypeR)

# Example data
data("BreastSubtypeRobj")
data("OSLO2EMIT0obj")

## ----map and prepare, eval=TRUE-----------------------------------------------
# Pre-processing: automatically apply tailored normalization, map probes/IDs to Entrez,
# and (optionally) impute missing values
data_input <- Mapping(
    OSLO2EMIT0obj$se_obj,
    method = "max", # mapping strategy (example)
    RawCounts = FALSE,
    impute = TRUE,
    verbose = FALSE
)

## ----multi-method run, eval=TRUE----------------------------------------------
methods <- c("parker.original", "PCAPAM50", "sspbc")

res <- BS_Multi(
    data_input = data_input,
    methods = methods,
    Subtype = FALSE,
    hasClinical = FALSE
)

# Per-sample calls (methods × samples)
head(res$res_subtypes, 5)

## ----auto mode and visualization, eval=TRUE-----------------------------------
res_auto <- BS_Multi(
    data_input = data_input,
    methods = "AUTO",
    Subtype = FALSE,
    hasClinical = FALSE
)

# visualize multi-method output and concordance
Vis_Multi(res_auto$res_subtypes)

## ----pam50 subtyping, eval=TRUE-----------------------------------------------
res_pam <- BS_parker(
    se_obj = data_input$se_NC, # object prepared for NC-based methods
    calibration = "Internal",
    internal = "medianCtr",
    Subtype = FALSE,
    hasClinical = FALSE
)

## ----aims subtyping, eval=TRUE------------------------------------------------
res_aims <- BS_AIMS(data_input$se_SSP)

## ----launch shiny, eval = FALSE-----------------------------------------------
# BreastSubtypeR::iBreastSubtypeR() # interactive GUI (local)

## ----installui dependencies, eval = FALSE-------------------------------------
# install.packages(c("shiny", "bslib"))

## ----load example data, eval = FALSE------------------------------------------
# exdir <- system.file("RshinyTest", package = "BreastSubtypeR")
# gex <- file.path(exdir, "OSLO2EMIT0_GEX_log2.FPKM.txt")
# clin <- file.path(exdir, "OSLO2EMIT0_clinical.txt")
# anno <- file.path(exdir, "OSLO2EMIT0_anno.txt")
# file.exists(gex)
# file.exists(clin)
# file.exists(anno)

## ----session info-------------------------------------------------------------
sessionInfo()

