#' mutFilterType
#' @description Filter variants based on variant types
#'
#' @param maf An MAF data frame, generated by \code{\link{vcfToMAF}} function.
#' @param keepType A group of variant classifications will be kept,
#' including 'exonic', 'nonsynonymous' and 'all'. Default: 'exonic'.
#' @importFrom methods is
#'
#' @return An MAF data frame where some variants
#' has T tag in CaTag column for variant type filtration
#'
#' @export mutFilterType
#' @examples
#' maf <- vcfToMAF(system.file("extdata",
#' "WES_EA_T_1_mutect2.vep.vcf", package="CaMutQC"))
#' mafF <- mutFilterType(maf)


mutFilterType <- function(maf, keepType = 'exonic') {
    # check user input
    if (!(is(maf, "data.frame"))) {
      stop("maf input should be a data frame, did you get it from vcfToMAF function?")
    }
  
    # process keepType
    if (keepType == 'exonic') {
      filterType <- c('RNA', 'Intron', 'IGR', '5\'Flank', '3\'Flank',
                 '5\'UTR', '3\'UTR')
    } else if (keepType == 'nonsynonymous') {
      filterType <- c("3'UTR", "5'UTR", "3'Flank", "Targeted_Region", "Silent",
                 "Intron", "RNA", "IGR", "Splice_Region", "5'Flank", "lincRNA",
                 "De_novo_Start_InFrame", "De_novo_Start_OutOfFrame",
                 "Start_Codon_Ins", "Start_Codon_SNP", "Stop_Codon_Del")
    } else if (keepType == 'all'){
      return(maf)
    } else {
      stop('Please select one category from exonic, nonsynonymous and all.')
    }
    # add T tag
    tags <- rownames(maf[maf$Variant_Classification %in% filterType, ])
    maf[tags, 'CaTag'] <- paste0(maf[tags, 'CaTag'], 'T')
    return(maf)
}
