% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeDMRs.R
\name{computeDMRs}
\alias{computeDMRs}
\title{Compute DMRs}
\usage{
computeDMRs(
  methylationData1,
  methylationData2,
  regions = NULL,
  context = "CG",
  method = "noise_filter",
  windowSize = 100,
  kernelFunction = "triangular",
  lambda = 0.5,
  binSize = 100,
  test = "fisher",
  pValueThreshold = 0.01,
  minCytosinesCount = 4,
  minProportionDifference = 0.4,
  minGap = 200,
  minSize = 50,
  minReadsPerCytosine = 4,
  parallel = FALSE,
  BPPARAM = NULL,
  cores = NULL
)
}
\arguments{
\item{methylationData1}{the methylation data in condition 1
(see \code{\link{methylationDataList}}).}

\item{methylationData2}{the methylation data in condition 2
(see \code{\link{methylationDataList}}).}

\item{regions}{a \code{\link{GRanges}} object with the regions where to 
compute the DMRs. If \code{NULL}, the DMRs are computed genome-wide.}

\item{context}{the context in which the DMRs are computed (\code{"CG"}, 
\code{"CHG"} or \code{"CHH"}).}

\item{method}{the method used to compute the DMRs (\code{"noise_filter"}, 
\code{"neighbourhood"} or \code{"bins"}). The \code{"noise_filter"} method 
uses a triangular kernel to smooth the number of reads and then performs a 
statistical test to determine which regions dispay different levels of 
methylation in the two conditions. The \code{"neighbourhood"} method 
computates differentially methylated cytosines. Finally, the \code{"bins"} 
method partiones the genome into equal sized tilling bins and performs the 
statistical test between the two conditions in each bin. For all three 
methods, the cytosines or bins are then merged into DMRs without affecting 
the inital parameters used when calling the differentiall methylated 
cytosines/bins (p-value, difference in methylation levels, minimum number of 
reads per cytosine).}

\item{windowSize}{the size of the triangle base measured in nucleotides. 
This parameter is required only if the selected method is 
\code{"noise_filter"}.}

\item{kernelFunction}{a \code{character} indicating which kernel function to 
be used. Can be one of \code{"uniform"}, \code{"triangular"}, 
\code{"gaussian"} or \code{"epanechnicov"}. This is required only if the 
selected method is \code{"noise_filter"}.}

\item{lambda}{numeric value required for the Gaussian filter 
(\code{K(x) = exp(-lambda*x^2)}). This is required only if the selected 
method is \code{"noise_filter"} and the selected kernel function is 
\code{"gaussian"}.}

\item{binSize}{the size of the tiling bins in nucleotides. This parameter is 
required only if the selected method is \code{"bins"}.}

\item{test}{the statistical test used to call DMRs (\code{"fisher"} for 
Fisher's exact test or \code{"score"} for Score test).}

\item{pValueThreshold}{DMRs with p-values (when performing the statistical 
test; see \code{test}) higher or equal than \code{pValueThreshold} are 
discarded. Note that we adjust the p-values using the Benjamini and 
Hochberg's method to control the false discovery rate.}

\item{minCytosinesCount}{DMRs with less cytosines in the specified context 
than \code{minCytosinesCount} will be discarded.}

\item{minProportionDifference}{DMRs where the difference in methylation 
proportion between the two conditions is lower than 
\code{minProportionDifference} are discarded.}

\item{minGap}{DMRs separated by a gap of at least \code{minGap} are not 
merged. Note that only DMRs where the change in methylation is in the same 
direction are joined.}

\item{minSize}{DMRs with a size smaller than \code{minSize} are discarded.}

\item{minReadsPerCytosine}{DMRs with the average number of reads lower than 
\code{minReadsPerCytosine} are discarded.}

\item{parallel}{Logical; run in parallel if \code{TRUE}.}

\item{BPPARAM}{A \code{BiocParallelParam} object controlling parallel execution.
This value will automatically set when parallel is \code{TRUE}, also able to set as manually.}

\item{cores}{Integer number of workers (must not exceed BPPARAM$workers).
This value will automatically set as the maximum number of system workers,
also able to set as manually.}
}
\value{
the DMRs stored as a \code{\link{GRanges}} object with the following 
metadata columns:
\describe{
 \item{direction}{a number indicating whether the region lost (-1)  or gain 
 (+1) methylation in condition 2 compared to condition 1.}
 \item{context}{the context in which the DMRs was computed (\code{"CG"}, 
 \code{"CHG"} or \code{"CHH"}).}
 \item{sumReadsM1}{the number of methylated reads in condition 1.}
 \item{sumReadsN1}{the total number of reads in condition 1.} 
 \item{proportion1}{the proportion methylated reads in condition 1.} 
 \item{sumReadsM2}{the number of methylated reads in condition 2.} 
 \item{sumReadsN2}{the total number reads in condition 2.}
 \item{proportion2}{the proportion methylated reads in condition 2.} 
 \item{cytosinesCount}{the number of cytosines in the DMR.} 
 \item{regionType}{a string indicating whether the region lost (\code{"loss"})  
 or gained (\code{"gain"}) methylation in condition 2 compared to condition 1.} 
 \item{pValue}{the p-value (adjusted to control the false discovery rate with 
 the Benjamini and Hochberg's method) of the statistical test when the DMR was 
 called.} 
}
}
\description{
This function computes the differentially methylated regions between two 
conditions.
}
\examples{

# load the methylation data
data(methylationDataList)

# the regions where to compute the DMRs
regions <- GRanges(seqnames = Rle("Chr3"), ranges = IRanges(1,1E5))

# compute the DMRs in CG context with noise_filter method
DMRsNoiseFilterCG <- computeDMRs(methylationDataList[["WT"]], 
                     methylationDataList[["met1-3"]], regions = regions, 
                     context = "CG", method = "noise_filter", 
                     windowSize = 100, kernelFunction = "triangular",  
                     test = "score", pValueThreshold = 0.01, 
                     minCytosinesCount = 4, minProportionDifference = 0.4, 
                     minGap = 200, minSize = 50, minReadsPerCytosine = 4, 
                     cores = 1)

\dontrun{
# compute the DMRs in CG context with neighbourhood method
DMRsNeighbourhoodCG <- computeDMRs(methylationDataList[["WT"]], 
                       methylationDataList[["met1-3"]], regions = regions, 
                       context = "CG", method = "neighbourhood", 
                       test = "score", pValueThreshold = 0.01, 
                       minCytosinesCount = 4, minProportionDifference = 0.4, 
                       minGap = 200, minSize = 50, minReadsPerCytosine = 4, 
                       cores = 1)

# compute the DMRs in CG context with bins method
DMRsBinsCG <- computeDMRs(methylationDataList[["WT"]], 
               methylationDataList[["met1-3"]], regions = regions, 
               context = "CG", method = "bins", binSize = 100, 
               test = "score", pValueThreshold = 0.01, minCytosinesCount = 4, 
               minProportionDifference = 0.4, minGap = 200, minSize = 50, 
               minReadsPerCytosine = 4, cores = 1)

}
}
\seealso{
\code{\link{filterDMRs}}, \code{\link{mergeDMRsIteratively}}, 
\code{\link{analyseReadsInsideRegionsForCondition}} and 
\code{\link{DMRsNoiseFilterCG}}
}
\author{
Nicolae Radu Zabet and Jonathan Michael Foonlan Tsang
}
