% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods-BulkPipeline.R
\name{BulkPipeline}
\alias{BulkPipeline}
\title{Pipeline for bulk long read RNA-seq data processing}
\usage{
BulkPipeline(
  config_file,
  outdir,
  fastq,
  annotation,
  genome_fa,
  genome_mmi,
  minimap2,
  samtools,
  controllers
)
}
\arguments{
\item{config_file}{Path to the JSON configuration file. See \code{\link{create_config}} for creating one.}

\item{outdir}{Path to the output directory. If it does not exist, it will be created.}

\item{fastq}{Path to the FASTQ file or a directory containing FASTQ files. Each file
will be processed as an individual sample.}

\item{annotation}{The file path to the annotation file in GFF3 / GTF format.}

\item{genome_fa}{The file path to the reference genome in FASTA format.}

\item{genome_mmi}{(optional) The file path to minimap2's index reference genome.}

\item{minimap2}{(optional) The path to the minimap2 binary. If not provided, FLAMES will
use a copy from bioconda via \code{basilisk}.}

\item{samtools}{(optional) The path to the samtools binary. If not provided, FLAMES will
use a copy from bioconda via \code{basilisk}.}

\item{controllers}{(optional, \strong{experimental}) A \code{crew_class_controller} object for running certain steps}
}
\value{
A \code{FLAMES.Pipeline} object. The pipeline could be run using \code{\link{run_FLAMES}}, and / or resumed using \code{\link{resume_FLAMES}}.
}
\description{
Semi-supervised isofrom detection and annotation for long read data.
This variant is meant for bulk samples. Specific parameters can be configured in
the config file (see \code{\link{create_config}}), input files are specified via
arguments.
}
\details{
By default FLAMES use minimap2 for read alignment. After the genome alignment step (\code{do_genome_align}), FLAMES summarizes the alignment for each read by grouping reads
with similar splice junctions to get a raw isoform annotation (\code{do_isoform_id}). The raw isoform
annotation is compared against the reference annotation to correct potential splice site
and transcript start/end errors. Transcripts that have similar splice junctions
and transcript start/end to the reference transcript are merged with the
reference. This process will also collapse isoforms that are likely to be truncated
transcripts. If \code{isoform_id_bambu} is set to \code{TRUE}, \code{bambu::bambu} will be used to generate the updated annotations.
Next is the read realignment step (\code{do_read_realign}), where the sequence of each transcript from the update annotation is extracted, and
the reads are realigned to this updated \code{transcript_assembly.fa} by minimap2. The
transcripts with only a few full-length aligned reads are discarded.
The reads are assigned to transcripts based on both alignment score, fractions of
reads aligned and transcript coverage. Reads that cannot be uniquely assigned to
transcripts or have low transcript coverage are discarded. The UMI transcript
count matrix is generated by collapsing the reads with the same UMI in a similar
way to what is done for short-read scRNA-seq data, but allowing for an edit distance
of up to 2 by default. Most of the parameters, such as the minimal distance to splice site and minimal percentage of transcript coverage
can be modified by the JSON configuration file (\code{config_file}).
}
\examples{
outdir <- tempfile()
dir.create(outdir)
# simulate 3 samples via sampling
reads <- ShortRead::readFastq(
  system.file("extdata", "fastq", "musc_rps24.fastq.gz", package = "FLAMES")
)
dir.create(file.path(outdir, "fastq"))
ShortRead::writeFastq(reads[1:100],
  file.path(outdir, "fastq/sample1.fq.gz"),
  mode = "w", full = FALSE
)
reads <- reads[-(1:100)]
ShortRead::writeFastq(reads[1:100],
  file.path(outdir, "fastq/sample2.fq.gz"),
  mode = "w", full = FALSE
)
reads <- reads[-(1:100)]
ShortRead::writeFastq(reads,
  file.path(outdir, "fastq/sample3.fq.gz"),
  mode = "w", full = FALSE
)
# prepare the reference genome
genome_fa <- file.path(outdir, "rps24.fa")
R.utils::gunzip(
  filename = system.file("extdata", "rps24.fa.gz", package = "FLAMES"),
  destname = genome_fa, remove = FALSE
)
ppl <- BulkPipeline(
  fastq = c(
    "sample1" = file.path(outdir, "fastq", "sample1.fq.gz"),
    "sample2" = file.path(outdir, "fastq", "sample2.fq.gz"),
    "sample3" = file.path(outdir, "fastq", "sample3.fq.gz")
  ),
  annotation = system.file("extdata", "rps24.gtf.gz", package = "FLAMES"),
  genome_fa = genome_fa,
  config_file = create_config(outdir, type = "sc_3end", threads = 1, no_flank = TRUE),
  outdir = outdir
)
ppl <- run_FLAMES(ppl) # run the pipeline
experiment(ppl) # get the result as SummarizedExperiment

}
\seealso{
\code{\link{create_config}} for creating a configuration file,
\code{\link{SingleCellPipeline}} for single cell pipelines,
\code{\link{MultiSampleSCPipeline}} for multi sample single cell pipelines.
}
