% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/G4VarImpact.R
\name{G4VarImpact}
\alias{G4VarImpact}
\title{Evaluate the Impact of Variants (SNVs, Indels, MNVs) on G4 Sequences}
\usage{
G4VarImpact(
  G4 = NULL,
  variants = NULL,
  ref_col = NULL,
  alt_col = NULL,
  mode = "s",
  sampleid_col = NULL
)
}
\arguments{
\item{G4}{A \code{GRanges} object representing the G4 regions. This object
must include a \code{sequence} metadata column containing the G4 sequences.
It should come from the output of the \code{G4HunterDetect} function with
\code{include_sequences = TRUE}.}

\item{variants}{A \code{GRanges} object representing the variants. This
object must include metadata columns for reference and alternative alleles.}

\item{ref_col}{A \code{character} string specifying the name of the column
in \code{variants} that contains the reference alleles.
Default is \code{"ref"}.}

\item{alt_col}{A \code{character} string specifying the name of the column
in \code{variants} that contains the alternative alleles.
Default is \code{"alt"}.}

\item{mode}{A \code{character} string indicating the mode of operation.
Set to \code{"s"} to evaluate the impact of individual variants on G4
regions one at a time (single-variant mode).
Set to \code{"m"} to assess the combined impact of multiple variants that
overlap the same G4 region within a sample (multi-variant Mode).
If using \code{"m"} mode, you must specify \code{sampleid_col}.}

\item{sampleid_col}{A \code{character} string specifying the name of the
column in \code{variants} that contains the sample IDs.
Required when \code{mode} is \code{"m"};
ignored if \code{mode} is \code{"s"}.}
}
\value{
A \code{GRanges} object with variant impact results:
\describe{
  \item{Mode "s" (single-variant mode )}{
    For each variant-G4 overlap:
    \itemize{
      \item Original G4 metadata (\code{G4.info.*})
      \item Variant information (\code{variant.info.*})
      \item Mutated sequence (\code{mutated.G4.seq})
      \item Annotated mutation sequence (\code{mutated.G4.anno.seq})
      \item New G4Hunter max_score (\code{mutated.max_score})
      \item Score difference (\code{score.diff})
    }
  }
  \item{Mode "m" (multi-variant mode)}{
    For each sample-G4 combination:
    \itemize{
      \item Original G4 metadata (\code{G4.info.*})
      \item Combined variant information (\code{variant.info.*})
      \item Mutated sequence with all variants incorporated
      (\code{mutated.G4.seq})
      \item Annotated mutation sequence (\code{mutated.G4.anno.seq})
      \item New G4Hunter max_score (\code{mutated.max_score})
      \item Score difference (\code{score.diff})
    }
  }
}
}
\description{
This function evaluates the impact of variants (SNVs, indels, and MNVs) on
G4 formation.
}
\examples{

if (!requireNamespace("BiocManager", quietly = TRUE)) {
  install.packages("BiocManager")
}

if (!requireNamespace("GenomicRanges", quietly = TRUE)) {
  BiocManager::install("GenomicRanges")
}

if (!requireNamespace("BSgenome.Hsapiens.UCSC.hg19", quietly = TRUE)) {
  BiocManager::install("BSgenome.Hsapiens.UCSC.hg19")
}

library(GenomicRanges)
library(BSgenome.Hsapiens.UCSC.hg19)

# Load sequence for chromosome 21 (hg19)
hg19 <- BSgenome.Hsapiens.UCSC.hg19
chr21_seq <- DNAStringSet(hg19$chr21)
# Chromosome name is needed
names(chr21_seq) <- "chr21"

# Detect G4s in human chromosome 21
G4 <- G4HunterDetect(chr21_seq)

# Load variants
data(snv_gr)

# 's' mode; single-variant mode ('s')
# evaluating each variant individually.
res_snv_s <- G4VarImpact(G4,
                         snv_gr,
                         ref_col = "ref",
                         alt_col = "alt")
print(res_snv_s)

# 'm' mode; multi-variant mode ('m')
# evaluating the combined impact of variants on G4s.
# Grouped by the sample IDs specified in the 'sampleid_col' column.
res_snv_m <- G4VarImpact(G4,
                         snv_gr,
                         ref_col = "ref",
                         alt_col = "alt",
                         mode = "m",
                         sampleid_col = "sampleid"
)
print(res_snv_m)
}
\seealso{
\code{\link{G4HunterDetect}} for detecting the G4 sequences in a
               given \code{DNAStringSet} object.
         \code{\link{filterVarImpact}} for filtering out variants with
               significant impact.
}
