#' getBins
#' @details Get uniformly-sized bins of specified width
#' This is a helper function used to generate binned data for \code{plotOnIdeo()}. It takes the chromosome-wide extents from \emph{ideo}, which is essentially the \emph{cytoBandIdeo} table from UCSC browser with the header as the first row. A use case is to generate bins using this function and supply the output to \code{avgByBin()} to bin the data.
#' @usage getBins(chroms, ideo, binLim = NULL, stepSize)
#' @param chroms (character) chromosomes to generate bins for
#' @param ideo (data.frame) ideogram table as generated by \code{getIdeo()}. See that function for details. 
#' @param binLim}{(numeric, length 2) [start, end] of genomic range to generate bins for. A value of NULL results in binning of entire chromosome
#' @param stepSize (integer) bin size in bases
#' @examples ideo_hg19 <- getIdeo("hg19")
#' chrom_bins <- getBins(c("chr1","chr2","chrX"), ideo_hg19,stepSize=5*100*1000)
#' @return (GRanges) bin ranges in 1-base coordinates
#' @seealso \code{getIdeo()},\code{avgByBin()}
#' @export
getBins <-function(chroms, ideo, binLim=NULL,stepSize)  {
    # make intervals for averaging
    if (missing(chroms)) chroms <- unique(ideo[,1])
    chromSteps <- cbind(CHROM=NA, START=NA,END=NA)
    
    for (chr in chroms) {
        csize <- max(ideo$chromEnd[ideo$chrom==chr])
        bstart <- seq(1,csize, stepSize)
        bend <- bstart + (stepSize-1); bend[length(bend)] <- csize
        df <- cbind(chr, bstart,bend)
        chromSteps <- rbind(chromSteps,df)

    }
    chromSteps <- chromSteps[-1,]
    full_chromInt <- GRanges(seqnames=chromSteps[,1], 
        ranges=IRanges(start=as.numeric(chromSteps[,2]),
        end=as.numeric(chromSteps[,3])))
    
    return(full_chromInt)
}
