% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/annotation_upset_chart_class.R
\name{upset_min_size}
\alias{upset_min_size}
\alias{upset_min_groups}
\alias{upset_max_groups}
\alias{upset_intersections}
\alias{upset_filters}
\title{UpSet chart filter helper functions}
\usage{
upset_min_size(min_size)

upset_min_groups(min_groups)

upset_max_groups(max_groups)

upset_intersections(combinations)
}
\arguments{
\item{min_size}{\code{numeric} The minimum number of items in an intersection}

\item{min_groups}{\code{numeric} The minimum number of groups in an intersection}

\item{max_groups}{\code{numeric} The maximum number of groups in an intersection}

\item{combinations}{\code{character} Vector of specific intersection combinations to include (e.g., c("A/B", "B/C"))}
}
\value{
A function that takes \code{region_data} as input and returns a logical
vector indicating which intersections to keep.
}
\description{
These functions create filters for the \code{annotation_upset_chart} class to
control which intersections are displayed in UpSet plots. Each function
returns a filter function that can be used with the \code{filter} parameter.
}
\details{
These filter functions work by analyzing the region data from the Venn
diagram to determine which intersections meet the specified criteria:
\itemize{
\item \code{upset_min_size()}: Filters intersections based on the number of items
\item \code{upset_min_groups()}: Filters intersections based on the minimum number
of groups involved
\item \code{upset_max_groups()}: Filters intersections based on the maximum number
of groups involved
\item \code{upset_intersections()}: Filters to show only specific intersection
combinations (e.g., "A/B", "B/C", "A/B/C")
}

For complex filtering logic, create custom filter functions:

\preformatted{
# Single filter
filter = upset_min_size(5)

# Specific combinations only
filter = upset_intersections(c("A/B", "B/C"))

# Custom filter function with AND logic
custom_filter <- function(region_data) {
  region_data$count >= 3 & region_data$count <= 10 & grepl("A", region_data$name)
}

# Custom filter function with OR logic
or_filter <- function(region_data) {
  region_data$count >= 5 | grepl("B/C", region_data$name)
}
}
}
\examples{
\dontrun{
# Filter to show only intersections with 5+ items
C <- annotation_upset_chart(factor_name = "V1", filter = upset_min_size(5))

# Filter to show only intersections involving 3+ groups
C <- annotation_upset_chart(factor_name = "V1", filter = upset_min_groups(3))

# Filter to show only intersections involving 2-4 groups (custom function)
group_range_filter <- function(region_data) {
  group_counts <- sapply(region_data$name, function(x) {
    if (x == "") return(0)
    groups <- strsplit(x, "/")[[1]]
    length(groups)
  })
  group_counts >= 2 & group_counts <= 4
}
C <- annotation_upset_chart(factor_name = "V1", filter = group_range_filter)

# Filter to show only specific combinations
C <- annotation_upset_chart(factor_name = "V1", 
                          filter = upset_intersections(c("A/B", "B/C")))

# Custom filter combining size and group criteria
size_and_group_filter <- function(region_data) {
  region_data$count >= 3 & sapply(region_data$name, function(x) {
    if (x == "") return(FALSE)
    groups <- strsplit(x, "/")[[1]]
    length(groups) >= 2
  })
}
C <- annotation_upset_chart(factor_name = "V1", filter = size_and_group_filter)
}

}
