test_that("AWS host URL parses correctly (aws_host)", {
  url <- "https://DOC-EXAMPLE-BUCKET1.s3.us-west-2.amazonaws.com/puppy.png"
  expect_silent(parsed <- Rarr:::.url_parse_aws(url))
  expect_identical(parsed$bucket, "DOC-EXAMPLE-BUCKET1")
  expect_identical(parsed$hostname, "https://s3.amazonaws.com")
  expect_identical(parsed$object, "puppy.png")
  expect_identical(parsed$region, "us-west-2")
})


test_that("AWS path URL parses correctly (aws_path)", {
  url <- "https://s3.us-west-2.amazonaws.com/DOC-EXAMPLE-BUCKET1/puppy.png"
  expect_silent(parsed <- .url_parse_aws(url))
  expect_identical(parsed$bucket, "DOC-EXAMPLE-BUCKET1")
  expect_identical(parsed$hostname, "https://s3.amazonaws.com")
  expect_identical(parsed$object, "puppy.png")
  expect_identical(parsed$region, "us-west-2")
})

test_that("Other S3-like URLs parse correctly (EMBL and EO paths)", {
  paths <- list(
    embl_path = "https://s3.embl.de/rarr-testing/bz2.zarr/.zarray",
    eopf_path_1 = "https://objectstore.eodc.eu:2222/e05ab01a9d56408d82ac32d69a5aae2a:202505-s02msil2a/17/products/cpm_v256/S2A_MSIL2A_20250517T085541_N0511_R064_T35QKA_20250517T112203.zarr/",
    eopf_path_2 = "https://objects.eodc.eu:443/e05ab01a9d56408d82ac32d69a5aae2a:202506-s01siwgrh/03/products/cpm_v256/S1C_IW_GRDH_1SDV_20250603T053151_20250603T053216_002614_0056AB_EE86.zarr"
  )

  # EMBL
  expect_silent(parsed <- Rarr:::.url_parse_other(paths$embl_path))
  expect_identical(parsed$bucket, "rarr-testing")
  expect_identical(parsed$hostname, "https://s3.embl.de")
  expect_identical(parsed$object, "bz2.zarr/.zarray")
  expect_identical(parsed$region, "auto")

  # EOPF style path 1
  expect_silent(parsed <- .url_parse_other(paths$eopf_path_1))
  expect_identical(
    parsed$bucket,
    "e05ab01a9d56408d82ac32d69a5aae2a:202505-s02msil2a"
  )
  expect_identical(parsed$hostname, "https://objectstore.eodc.eu:2222")
  expect_identical(
    parsed$object,
    "17/products/cpm_v256/S2A_MSIL2A_20250517T085541_N0511_R064_T35QKA_20250517T112203.zarr/"
  )
  expect_identical(parsed$region, "auto")

  # EOPF style path 2
  expect_silent(parsed <- .url_parse_other(paths$eopf_path_2))
  expect_identical(
    parsed$bucket,
    "e05ab01a9d56408d82ac32d69a5aae2a:202506-s01siwgrh"
  )
  expect_identical(parsed$hostname, "https://objects.eodc.eu:443")
  expect_identical(
    parsed$object,
    "03/products/cpm_v256/S1C_IW_GRDH_1SDV_20250603T053151_20250603T053216_002614_0056AB_EE86.zarr"
  )
  expect_identical(parsed$region, "auto")
})

test_that("s3 provider is determined correctly", {
  paths <- list(
    aws_host = "https://DOC-EXAMPLE-BUCKET1.s3.us-west-2.amazonaws.com/puppy.png",
    aws_path = "https://s3.us-west-2.amazonaws.com/DOC-EXAMPLE-BUCKET1/puppy.png",
    embl_path = "https://s3.embl.de/rarr-testing/bz2.zarr/.zarray",
    eopf_path_1 = "https://objectstore.eodc.eu:2222/e05ab01a9d56408d82ac32d69a5aae2a:202505-s02msil2a/17/products/cpm_v256/S2A_MSIL2A_20250517T085541_N0511_R064_T35QKA_20250517T112203.zarr/",
    eopf_path_2 = "https://objects.eodc.eu:443/e05ab01a9d56408d82ac32d69a5aae2a:202506-s01siwgrh/03/products/cpm_v256/S1C_IW_GRDH_1SDV_20250603T053151_20250603T053216_002614_0056AB_EE86.zarr"
  )

  expect_identical(.determine_s3_provider(paths[["aws_host"]]), "aws")
  expect_identical(.determine_s3_provider(paths[["embl_path"]]), "other")
  expect_identical(.determine_s3_provider("https://foo.bar/baz"), "other")
  expect_identical(.determine_s3_provider(paths[["eopf_path_1"]]), "other")
  expect_identical(.determine_s3_provider(paths[["eopf_path_2"]]), "other")
})
