% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findMarkersTree.R
\name{findMarkersTree}
\alias{findMarkersTree}
\title{Generate marker decision tree from single-cell clustering output}
\usage{
findMarkersTree(
  features,
  class,
  oneoffMetric = c("modified F1", "pairwise AUC"),
  metaclusters,
  featureLabels,
  counts,
  celda,
  seurat,
  threshold = 0.9,
  reuseFeatures = FALSE,
  altSplit = TRUE,
  consecutiveOneoff = FALSE,
  autoMetaclusters = TRUE,
  seed = 12345
)
}
\arguments{
\item{features}{features-by-samples numeric matrix, e.g. counts matrix.}

\item{class}{Vector of cell cluster labels.}

\item{oneoffMetric}{A character string. What one-off metric to run, either
`modified F1` or `pairwise AUC`. Default is 'modified F1'.}

\item{metaclusters}{List where each element is a metacluster (e.g. known
cell type) and all the clusters within that metacluster (e.g. subtypes).}

\item{featureLabels}{Vector of feature assignments, e.g. which cluster
does each gene belong to? Useful when using clusters of features
(e.g. gene modules or Seurat PCs) and user wishes to expand tree results
to individual features (e.g. score individual genes within marker gene modules).}

\item{counts}{Numeric counts matrix. Useful when using clusters
of features (e.g. gene modules) and user wishes to expand tree results to
individual features (e.g. score individual genes within marker gene
modules). Row names should be individual feature names.}

\item{celda}{A \emph{celda_CG} or \emph{celda_C} object.
Counts matrix has to be provided as well.}

\item{seurat}{A seurat object. Note that the seurat functions
\emph{RunPCA} and \emph{FindClusters} must have been run on the object.}

\item{threshold}{Numeric between 0 and 1. The threshold for the oneoff
metric. Smaller values will result in more one-off splits. Default is 0.90.}

\item{reuseFeatures}{Logical. Whether or not a feature can be used more than
once on the same cluster. Default is TRUE.}

\item{altSplit}{Logical. Whether or not to force a marker for clusters that
are solely defined by the absence of markers. Default is TRUE.}

\item{consecutiveOneoff}{Logical. Whether or not to allow one-off splits at
consecutive brances. Default is FALSE.}

\item{autoMetaclusters}{Logical. Whether to identify metaclusters prior to
creating the tree based on the distance between clusters in a UMAP 
dimensionality reduction projection. A metacluster is simply a large
cluster that includes several clusters within it. Default is TRUE.}

\item{seed}{Numeric. Seed used to enable reproducible UMAP results
for identifying metaclusters. Default is 12345.}
}
\value{
A named list with six elements:
\itemize{
  \item rules - A named list with one data frame for every label. Each
 data frame has five columns and gives the set of rules for disinguishing
 each label.
  \itemize{
   \item feature - Marker feature, e.g. marker gene name.
   \item direction - Relationship to feature value. -1 if cluster is
   down-regulated for this feature, 1 if cluster is up-regulated.
   \item stat - The performance value returned by the splitting metric for
 this split.
   \item statUsed - Which performance metric was used. "Split" if information
 gain and "One-off" if one-off.
   \item level - The level of the tree at which is rule was defined. 1 is the
 level of the first split of the tree.
   \item metacluster - Optional. If metaclusters were used, the metacluster
    this rule is applied to.
  }
 \item dendro - A dendrogram object of the decision tree output. Plot with 
 plotDendro()
 \item classLabels - A vector of the class labels used in the model, i.e.
  cell cluster labels.
 \item metaclusterLabels - A vector of the metacluster labels
  used in the model
 \item prediction - A character vector of label of predictions of the
 training data using the final model. "MISSING" if label prediction was
 ambiguous.
 \item performance - A named list denoting the training performance of the
 model:
 \itemize{
  \item accuracy - (number correct/number of samples) for the whole set of
 samples.
  \item balAcc - mean sensitivity across all clusters
  \item meanPrecision - mean precision across all clusters
  \item correct - the number of correct predictions of each cluster
  \item sizes - the number of actual counts of each cluster
  \item sensitivity - the sensitivity of the prediciton of each cluster
  \item precision - the precision of the prediciton of each cluster
 }
}
}
\description{
Create a decision tree that identifies gene markers for given
 cell populations. The algorithm uses a decision tree procedure to generate
 a set of rules for each cell cluster defined by single-cell clustering.
 Splits are determined by one of two metrics at each split: a one-off metric
 to determine rules for identifying clusters by a single feature, and a
 balanced metric to determine rules for identifying sets of similar clusters.
}
\examples{
# Generate simulated single-cell dataset using celda 
sce <- celda::simulateCells("celda_CG", K = 4, L = 10, G = 100)

# Select top features
sce <- selectFeatures(sce)

# Celda clustering into 5 clusters & 10 modules
sce <- celda_CG(sce, K=5, L=10, verbose=FALSE)

# Get features matrix and cluster assignments
factorizedCounts <- factorizeMatrix(sce, type = "counts")
featureMatrix <- factorizedCounts$counts$cell
classes <- as.integer(celdaClusters(sce))

# Generate Decision Tree
DecTree <- findMarkersTree(featureMatrix, classes)

# Plot dendrogram
plotDendro(DecTree)

}
