% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multisplit.R
\name{multisplit}
\alias{multisplit}
\title{Multi-sample splitting}
\usage{
multisplit(x, y, clvar = NULL, B = 50, proportion.select = 1/6,
  standardize = FALSE, family = c("gaussian", "binomial"),
  parallel = c("no", "multicore", "snow"), ncpus = 1L, cl = NULL,
  check.input = TRUE)
}
\arguments{
\item{x}{a matrix or list of matrices for multiple data sets. The matrix or
matrices have to be of type numeric and are required to have column names
/ variable names. The rows and the columns represent the observations and
the variables, respectively.}

\item{y}{a vector, a matrix with one column, or list of the aforementioned
objects for multiple data sets. The vector, vectors, matrix, or matrices
have to be of type numeric. For \code{family = "binomial"}, the response
is required to be a binary vector taking values 0 and 1.}

\item{clvar}{a matrix or list of matrices of control variables.}

\item{B}{number of sample splits.}

\item{proportion.select}{proportion of variables to be selected by Lasso in
the multi-sample splitting step.}

\item{standardize}{a logical value indicating whether the variables should be
standardized.}

\item{family}{a character string naming a family of the error distribution;
either \code{"gaussian"} or \code{"binomial"}.}

\item{parallel}{type of parallel computation to be used. See the 'Details' section.}

\item{ncpus}{number of processes to be run in parallel.}

\item{cl}{an optional \strong{parallel} or \strong{snow} cluster used if
\code{parallel = "snow"}. If not supplied, a cluster on the local machine is created.}

\item{check.input}{a logical value indicating whether the function should
check the input. This argument is used to call
\code{\link{multisplit}} within
\code{\link{test_hierarchy}}.}
}
\value{
The returned value is an object of class \code{"hierM"}, consisting
of a list with number of elements corresponding to the number of data sets.
Each element (corresponding to a data set
% with \code{nobs} observations) contains a list with two matrices.
The first matrix
% of size \code{B x [nobs / 2]}
contains the indices of the second half of variables (which were not used
to select the variables). The second matrix
% of size \code{B x [nobs * proportion.select]}
contains the column names / variable names of the selected variables.
}
\description{
The data is randomly split in two halves w.r.t. the observations and
variable selection using Lasso is performed on one half. Whereas the second
half and the selected variables are later used for testing by the function
\code{\link{test_only_hierarchy}}. This is repeated multiple times.
}
\details{
A given data with \code{nobs} is randomly split in two halves w.r.t.
the observations and \code{nobs * proportion.select} variables are selected
using Lasso (implemented in \code{\link{glmnet}}) on one half.
Control variables are not penalized if supplied
using the argument \code{clvar}. This is repeated \code{B} times for each
data set if multiple data sets are supplied. Those splits (i.e. second
halves of observations) and corresponding selected variables are used to
perform hierarchical testing by the function
\code{\link{test_only_hierarchy}}.

The multi-sample split step can be run in parallel across the different
sample splits (\code{B} corresponds to number of sample splits) by
specifying the arguments \code{parallel} and \code{ncpus}.
There is an optional argument \code{cl} if \code{parallel = "snow"}.
There are three possibilities to set the argument \code{parallel}:
\code{parallel = "no"} for serial evaluation (default),
\code{parallel = "multicore"} for parallel evaluation
using forking, and \code{parallel = "snow"} for parallel evaluation
using a parallel socket cluster. It is recommended to select
\code{\link{RNGkind}("L'Ecuyer-CMRG")} and set a seed to ensure that
the parallel computing of the package \code{hierinf} is reproducible.
This way each processor gets a different substream of the pseudo random
number generator stream which makes the results reproducible if the arguments
(as \code{sort.parallel} and \code{ncpus}) remain unchanged. See the vignette
or the reference for more details.
}
\examples{
n <- 200
p <- 500
library(MASS)
set.seed(3)
x <- mvrnorm(n, mu = rep(0, p), Sigma = diag(p))
colnames(x) <- paste0("Var", 1:p)
beta <- rep(0, p)
beta[c(5, 20, 46)] <- 1
y <- x \%*\% beta + rnorm(n)

set.seed(84)
res.multisplit <- multisplit(x = x, y = y, family = "gaussian")

}
\references{
Renaux, C. et al. (2018), Hierarchical inference for genome-wide
association studies: a view on methodology with software. (arXiv:1805.02988)

Meinshausen, N., Meier, L. and Buhlmann, P. (2009), P-values for
high-dimensional regression, Journal of the American Statistical Association
104, 1671-1681.
}
\seealso{
\code{\link{cluster_var}},
\code{\link{cluster_position}},
\code{\link{test_only_hierarchy}},
\code{\link{test_hierarchy}}, and
\code{\link{compute_r2}}.
}
