% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/selectAnchors.R
\name{selectAnchors}
\alias{selectAnchors}
\title{Select Anchors for Nonlinear RT Model}
\usage{
selectAnchors(
  object,
  useID = FALSE,
  tolmz = 0.003,
  tolQ = 0.3,
  tolrtq = 0.3,
  windx = 0.03,
  windy = 0.03,
  brackets_ignore = c("(", "[", "{")
)
}
\arguments{
\item{object}{metabCombiner object.}

\item{useID}{logical. Option to first search for IDs as anchors.}

\item{tolmz}{numeric. m/z tolerance for prospective anchors}

\item{tolQ}{numeric. Quantile Q tolerance for prospective anchors}

\item{tolrtq}{numeric. Linear RT quantile tolerance for prosepctive anchors.}

\item{windx}{numeric. Retention time exclusion window around each anchor in
X dataset. Optimal values are between 0.01 and 0.05 min (1-3s)}

\item{windy}{numeric. Retention time exclusion window around each anchor in
dataset Y. Optimal values are between 0.01 and 0.05 min (1-3s)}

\item{brackets_ignore}{If useID = TRUE, bracketed identity strings of the
types included in this argument will be ignored.}
}
\value{
\code{metabCombiner} object with updated \code{anchors} slot. This
    is a data.frame of feature pairs that shall be used to map between
    retention times using a GAM or LOESS model.

\item{idx}{identities of features from dataset X}
\item{idy}{identities of features from dataset Y}
\item{mzx}{m/z values of features from dataset X}
\item{mzy}{m/z values of features from dataset Y}
\item{rtx}{retention time values of features from dataset X}
\item{rty}{retention time values of features from dataset Y}
\item{rtProj}{model-projected retention time values from X to Y}
\item{Qx}{abundance quantile values of features from dataset X}
\item{Qy}{abundance quantile values of features from dataset Y}
\item{adductX}{adduct label of features from dataset X}
\item{adductY}{adduct label of features from dataset Y}
\item{group}{m/z feature group of feature pairing}
\item{labels}{anchor labels; "I" for identity, "A" for normal anchors}
}
\description{
A subset of possible alignments in the \code{combinedTable} are used as
    ordered pairs to anchor a retention time projection model. Alignments of
    abundant features are prominent targets for anchor selection, but shared
    identified features (i.e. feature pairs where idx = idy) may be used.
}
\details{
In order to map between two sets of retention times, a set of ordered pairs
    need to be selected for the spline fit. This function relies on mutually
    abundant features to select these ordered pairs. In iterative steps, the
    most abundant (as indicated by Q value) in one dataset is selected along
    with its counterpart, and all features within some retention time window
    specified by \code{windx} & \code{windy} arguments are excluded. This
    process is repeated until all features have been considered.

\code{tolQ} & \code{tolmz} arguments restrict to feature pairs that have
    differences in Q & m/z within these tolerances. \code{tolrtq} further
    limits to feature pairs those with relative differences in linear
    retention time quantiles, calculated as
    \eqn{rtqx = (rtx - min(rtx)) / (max(rtx) - min(rtx))} &
    \eqn{rtqy = (rty - min(rty)) / (max(rty) - min(rty))}

Shared identities (in which idx & idy columns have matching, non-empty &
    non-bracketed strings) may be used if \code{useID} is set to TRUE. In
    this case, shared identities will be searched first and will not be
    subject to any of the restrictions in m/z, Q, or rt. The iterative
    process proceeds after processing of shared identities.
}
\examples{
data(plasma30)
data(plasma20)

p30 <- metabData(plasma30, samples = "CHEAR")
p20 <- metabData(plasma20, samples = "Red", rtmax = 17.25)
p.comb <- metabCombiner(xdata = p30, ydata = p20, binGap = 0.005)

##example 1 (no known IDs used)
p.comb <- selectAnchors(p.comb, tolmz = 0.003, tolQ = 0.3, windx = 0.03,
    windy = 0.02, tolrtq = 0.3)

##example 2 (known IDs used)
p.comb <- selectAnchors(p.comb, useID = TRUE, tolmz = 0.003, tolQ = 0.3)

##To View Plot of Ordered Pairs
anchors = getAnchors(p.comb)
plot(anchors$rtx, anchors$rty, main = "Selected Anchor Ordered Pairs",
    xlab = "rtx", ylab = "rty")

}
\seealso{
\code{\link{getAnchors}}, \code{\link{fit_gam}}, \code{\link{fit_loess}}
}
