% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metagene.R
\name{metagene2}
\alias{metagene2}
\title{A class to manage metagene analysis.}
\format{
A metagene experiment manager
}
\value{
\code{metagene2$new} returns a \code{metagene2} object which contains the
normalized coverage values for every regions in all specified BAM files.
}
\description{
This metagene2 class encapsulates all of the steps necessary to perform
metagene analyses, which are aggregations of coverages over multiple regions 
(genes) to reveal patterns that might not be apparent from looking at
individual regions. It will allow to load, convert and normalize bam 
alignments and regions files/data. Once the data is ready, the user can then
choose to produce metagene plots on the data or some subset of it.
}
\details{
Most metagene analyses are a two-step affair:
\enumerate{
 \item Initialize the object using \code{mg = metagene2$new()}, specifying 
       which regions and bam files should be used for the analysis.
 \item Generate a metagene plot using \code{mg$produce_metagene}, specifying 
       any additional parameter (Number of bins, facetting variables, etc).
}

The \code{metagene2} object will then internally chain all 6 required  
processing steps, updating its internal caches along the way:
\enumerate{
 \item Coverages are inferred from bam files (\code{metagene2$new}).
 \item Coverages from multiple bam files are grouped and normalized 
      (\code{mg$group_coverages}).
 \item Coverages are binned together (\code{mg$bin_coverages})
 \item Binned coverages are split according to the type of region they belong 
       to (\code{mg$split_coverages_by_regions}).
 \item Coverage means and confidence intervals are calculated for each 
       region * group combination (\code{mg$calculate_ci}).
 \item Metadata is added to the calculated coverages ( 
       \code{mg$add_metadata}).
 \item The metagene is plotted (\code{mg$plot}).
}

Each of these steps has an associated function, which takes as input certain 
parameters of the metagene analysis and returns an intermediary structure
of interest (coverages, binned coverages, long-form data frame of confidence 
intervals, etc). Those are described below, in the "Processing methods" 
section.

All processing methods automatically call previous processing steps if
those have not already been run. For example, there is no need to call
\code{mg$group_coverages()} before calling \code{mg$bin_coverages()}: the
metagene2 object will automatically detect that certain prerequisite steps 
have not yet been performed, and run them.

Additionally, when calling produce_metagene a second time to change certain
analysis parameters after generating an initial metagene plot, only the 
required caches are reset: all non-impacted aspects of the analysis are left 
untouched, decreasing processing time.

For further examples, see the metagene2 vignette.
}
\section{Constructor}{


\strong{Usage:}

 \code{mg <- metagene2$new(regions, bam_files, padding_size = 0,
                           cores = SerialParam(), verbose = FALSE,
                           force_seqlevels = FALSE, paired_end = FALSE,
                           assay = 'chipseq', strand_specific=FALSE,
                           paired_end_strand_mode=2,
                           region_mode="auto", region_metadata=NULL, 
                           extend_reads=0, invert_strand=FALSE, ...)}

\strong{Description:}

This method returns a new \code{metagene2} object. Upon initialization, a  
\code{metagene2} object calculates coverages over all given regions in the 
provided bam files. Any and all parameter associated with any of the 
processing steps can be initialized upon object construction. All analysis  
parameters that are not explicitly specified in the constructor call are  
initialized to sensible defaults.

\strong{Parameters:}
\describe{
   \item{regions}{A description of all regions over which metagenes will be calculated.

                  When \code{region_mode} is "separate", those should be provided using
                  a \code{GRanges} object representing all individual, contiguous regions
                  to be examined.

                  When \code{region_mode} is "stitch", those should be provided using a 
                  \code{GRangesList} object where each individual \code{GRanges} element
                  represents a set of regions to be stitched together.

                  As a convenience, in "separate" mode, \code{metagene2} will convert any
                  passed in \code{GRangesList} into an unlisted \code{GRanges} with an 
                  additional \code{region_name} metadata column containing 
                  the name of the \code{GRangesList} element it was extracted 
                  from.

                  Also as a convenience, \code{regions} can also be a \code{character} 
                  \code{vector} of filenames, which are then imported into a GRangesList.
                  Supported file formats are BED, narrowPeak, broadPeak, gff and gtf.}
   \item{bam_files}{A \code{vector} of BAM filenames. The BAM files must be
                   indexed. i.e.: if a file is named file.bam, there must
                   be a file named file.bam.bai or file.bai in the same 
                   directory. If \code{bam_files} is a named vector, then the provided names
                   can be used downstream to refer to those bam files. If no
                   names are provided, \code{metagene2} will try to infer appropriate ones.}
   \item{assay}{\code{'chipseq'}, \code{'rnaseq'} or NULL. If non-NULL, metagene will
                set other parameters, such as region_mode and strand_specific, to logical
                values for the given assay. Default: \code{'chipseq'}}
   \item{region_mode}{Set the way the \code{regions} parameter is interpreted. Can be 
                      \code{'separate'}, \code{'stitch'} or \code{'auto'}. In separate mode,
                      \code{regions} is expected to be a GRanges defining individual, contiguous regions. In
                      \code{'stitch'} mode, \code{regions} is expected to be a GRangesList 
                      where each \code{GRanges} element represents a set of regions to be 
                      stitched together and treated as a single logical region. If \code{'auto'}
                      then a logical value is inferred from the \code{assay} parameter.
                      Default: \code{'auto'}}
   \item{region_metadata}{A data-frame of metadata to be associated with the elements of \code{regions}.
                          It must contain has many rows as there are elements in \code{regions}. If 
                          \code{region_metadata} is NULL but \code{regions} has an mcols element,
                          then it is used.}
   \item{padding_size}{The provided \code{regions} will be extended on each side by the
                       value of this parameter. The padding_size must be a
                       non-negative integer. Default = 0.}
   \item{cores}{The number of cores available to parallelize the analysis.
               Either a positive integer or a \code{BiocParallelParam}.
               Default: \code{SerialParam()}.}
   \item{verbose}{Print progression of the analysis. A logical constant.
                   Default: \code{FALSE}.}
   \item{force_seqlevels}{If \code{TRUE}, remove regions that are not found
               in bam file header. Default: \code{FALSE}. \code{TRUE} and \code{FALSE}
               respectively correspond to pruning.mode = "coarse"
               and "error" in ?seqinfo.}
   \item{paired_end}{Set this to \code{TRUE} if the provided bam files
                     describe paired-end reads. If \code{FALSE}, single-ended
                     data are expected. Default: \code{FALSE}}
   \item{strand_specific}{If \code{TRUE}, only reads which align to the same 
                          strand as those specified in \code{regions} will
                          count toward coverage for that region. Useful for RNA-seq
                          profiles generated from strand-specific libraries, such
                          as Illumina TruSeq. Default: \code{'FALSE'}}
   \item{paired_end_strand_mode}{\code{'1'} or \code{'2'}. In paired-end mode,
                                 indicates which read in a pair sets the pair's strand.
                                 If \code{1}, this is the first read (This should be used
                                 with directional protocols such as Directional Illumina 
                                 (Ligation) or Standard SOLiD).
                                 If \code{2}, this is the second read (This should be used
                                 with directional protocols such as dUTP, NSR, NNSR, 
                                 or Illumina stranded TruSeq PE).
                                 Ignored if either paired_end or strand_specific is FALSE.
                                 Default: \code{'2'}}
   \item{extend_reads}{Extend individual reads to have a minimum length equal to this parameter.
                       When set to 0, no read extension occurs. This is useful for single-end
                       chip-seq experiments, where the length of the captured fragment is usually
                       longer than the sequenced read.}
   \item{invert_strand}{If \code{TRUE}, coverages for the given regions will be inferred 
                        from the coverage on the strand opposite theirs. Useful
                        for single-end stranded experiments which use cDNA.
                        This parameter is ignored if strand-specific is \code{FALSE}.}
   \item{...}{Additional parameters for the metagene analysis. See \code{produce_metagene}
              for a list of possible parameters.}
}

   \code{metagene2$new} returns a \code{metagene2} object that contains the
       coverages for every BAM files in the regions from the \code{regions}
       parameter.
}

\section{produce_metagene()}{


\strong{Usage:}

\code{mg$produce_metagene(...)}

\strong{Description:}

\code{produce_metagene} is the workhorse method of the metagene2 object.
This method performs all of the necessary analysis steps for the production
of the metagene plot, and returns that plot. Any and all parameters of the 
metagene analysis, as documented in the individual processing steps, can be 
passed to \code{produce_metagene}. The metagene2 object will then determines
which intermediate caches would be affected by changes to those parameters,
invalidate them, and rerun all steps up to the plotting. This makes 
\code{produce_metagene} ideal for fast, iterative takes on the data.

Below we present those parameters and a brief description of their usage.
Please refer to the affected processing step for a more in-depth explanation
of each parameter.

\strong{Parameters:}
\describe{
   \item{design}{A \code{data.frame} that describes the grouping of the bam files
           into design groups. By default, each bam file is its own design group.
           See \code{group_coverages}.}
   \item{normalization}{The algorithm to use to normalize coverages,
                       \code{NULL} (no normalization) or "RPM". By default,
                       no normalization occurs. See \code{group_coverages}.}
   \item{design_filter}{Indices indicating which subset of design groups should 
                        be included in the analysis. By default, all design  
                        groups/bam files are included. See 
                        \code{group_coverages}.}
   \item{bin_count}{The number of bins regions should be split into. Defaults
                    to 100. See \code{bin_coverages}.}
   \item{region_filter}{The subset of regions to be kept for the analysis.
                        By default, all regions are kept. See \code{bin_coverages}}
   \item{split_by}{Which metadata columns should we use to split the set of 
                   regions into subset of interests? Defaults to "region_name",
                   an automatically added column. 
                   See \code{split_coverages_by_regions}.}
   \item{alpha}{The alpha level of the confidence interval estimates.
                Defaults to 0.05. See \code{calculate_ci}.}
   \item{sample_count}{The number of draws to perform in the bootstrap
                       calculations used to calculate the confidence inteval.  
                       Defaults to 1000. See \code{calculate_ci}}
   \item{resampling_strategy}{The resampling strategy to be used when performing the
                              bootstrap analysis, which can be either \code{'profile'}
                              or \code{'bin'}. Defaults to \code{'bin'}. See
                              \code{calculate_ci}.}
   \item{design_metadata}{A data-frame containing metadata for the design groups.
                          By default, no metadata is associated. See 
                          \code{add_metadata}.}
   \item{title}{A title to add to the graph. See \code{plot}.}
   \item{x_label}{X-axis label for the metagene plot. See \code{plot}.}
   \item{facet_by}{A formula to be used for facetting the metagene plot. 
                   By default, no facetting occurs. See \code{plot}.}
   \item{group_by}{The metadata column used to build the color scale. By 
                   default, the combination of design and region name is 
                   used. See \code{plot}.}
}
}

\section{Processing methods}{


Each of the following methods perform one step of metagene processing.
Most do not need to be called explicitly. Instead, you can simply call 
\code{produce_metagene}. However, you can use them to access intermediary
results: grouped coverages, binned coverages, split coverages, and long-form
data-frame of coverages with confidence intervals.
}

\section{group_coverages}{


\strong{Usage:}

 \code{mg$group_coverages(design=NA, normalization=NA, 
           design_filter=NA, simplify=FALSE)}

\strong{Description:}

 This method normalizes genome-wide coverages, then groups
 them according to the specified design groups. It returns
 a list of possible read orientations (+, -, *), each element
 of which is either NULL (depending on the value of the 
 strand_specific parameter) or a list of possible design groups.
 In turn, the lists of design groups contain lists of \code{Rle}
 objects representing coverage over a specific chromosome or sequence.

\strong{Parameters:}
\describe{
   \item{design}{A \code{data.frame} that describes the grouping of the bam files
           into design groups. The first column of the design should contain the 
           names of bam_files passed on initialization. Each subsequent columns
           represents a design group, that is to say a combination of bam files
           whose coverages should be grouped together into a logical unit.
           These columns should contain integer values indicating whether the 
           bam files on that row should be excluded (0), included as an`
           "input" (1) or included as a "control" (2) within the specified 
           design group. Control samples are used for "log2_ratio"
           normalization, but are ignored for no or "RPM" normalization.
           \code{NA} can be used keep previous design value. Default: \code{NA}.}
   \item{normalization}{The algorithm to use to normalize coverages. Possible
                       values are \code{NULL} (no normalization), "RPM" and "log2_ratio".
                       "RPM" transforms raw counts into Reads-Per-Million.
                       "log2_ratio" uses the formula log2((input RPM + 1) / (control RPM + 1))
                       to calculate a log-ratio between input and control. \code{NA} can 
                       be used keep the previous value. Default: \code{NA}}
   \item{design_filter}{A logical vector specifying which of the design groups specified
                        within the \code{design} parameter should be included in the metagene.
                        Useful for quickly reprocessing a subset of samples.
                        \code{NA} can be used keep previous design value. Default: \code{NA}}
   \item{simplify}{In single strand mode, set \code{simplify} to \code{TRUE} to return 
                   only the '*' coverage and omit the empty '+' and '-' components.
                   Default: \code{FALSE}}
}
}

\section{bin_coverages}{


\strong{Usage:}

 \code{mg$bin_coverages(bin_count=NA, region_filter=NA)}

\strong{Description:}

This method summarizes the coverage over regions of interests
into a specified number of bins. For each design group, it 
produces a matrix of binned coverages where each row represents a region,
and each column represents a bin. Those are returned in a named list where
each element contains the resulting matrix for a specific design group.

\strong{Parameters:}
\describe{
   \item{bin_count}{The number of bins regions should be split into. The specified 
                    bin_count must always be equal or higher than the minimum size of
                    the specified regions. \code{NA} can be used to keep the previous
                    value. Default: \code{NA}.}
   \item{region_filter}{This parameter defines the subset of regions within the \code{regions}
                        parameter passed on initialization on which the metagene
                        should be generated. \code{region_filter} can be (1) a quosure, to be evaluated
                        in the context of the \code{region_metadata} data-frame, (2) a character
                        vector containing the names of the regions to be used or (3) a logical or numeric
                        vector to be used for subsetting. \code{NA} can be used to keep the previous
                        value. Default: \code{NA}}
}
}

\section{split_coverages_by_regions}{


\strong{Usage:}

 \code{mg$split_coverages_by_regions(split_by=NA)}

\strong{Description:}

This methods splits the matrices generated by mg$bin_coverages
into groups of regions where the values of the metadata columns
specified by \code{split_by} are homogeneous. It returns a list
where each element represents a design group: each of those
element is in turn a list representing groups of regions for which
all metadata values specified by "split_by" are equal. The leaf elements
of this list hierarchy are coverage matrices where each row represents a
region, and each column represents a bin.

\strong{Parameters:}
\describe{
   \item{split_by}{A vector of column names from the region_metadata
                   parameter, as specified on metagene initialization. The 
                   selected columns must allow conversion into a factor.
                   By default, this is set to region_name, a metadata column
                   which is automatically generated by metagene. \code{NA} can 
                   be used to keep the previous value. Default: \code{NA}}
}
}

\section{calculate_ci}{


\strong{Usage:}

 \code{mg$calculate_ci(alpha = NA, sample_count = NA, resampling_strategy=NA)}

\strong{Description:}

This method calculates coverage means and confidence intervals for all 
design_group * region * bin combination. These are returned as a long-form
data-frame.

\strong{Parameters:}
\describe{
   \item{alpha}{The alpha level of the confidence interval estimate.
               \code{NA} can be used to keep the previous value. 
               Default: \code{NA}}
   \item{sample_count}{The number of draws to perform in the bootstrap
                       calculations used to calculate the confidence inteval.  
                       \code{NA} can be used to keep the previous value. Default: \code{NA}}
   \item{resampling_strategy}{The resampling strategy to be used when performing the
                              bootstrap analysis, which can be either \code{'profile'}
                              or \code{'bin'}. In \code{'profile'} mode, whole profiles
                              across all bins are resampled. In \code{'bin'} mode,
                              each bin is resampled individually and independantly from
                              all others. \code{NA} can be used to keep the previous value.
                              Default: \code{NA}}
}
}

\section{add_metadata}{


\strong{Usage:}

 \code{mg$add_metadata(design_metadata=NA)}

\strong{Description:}

This method adds design group and region metadata to the data-frame
produced by \code{mg$calculate_ci} for easier plotting.

\strong{Parameters:}
\describe{
   \item{design_metadata}{A data-frame containing metadata for the design groups.
                          It must contain as many rows as there are design groups,
                          and must contain at least one column named 'design'
                          which is used to match the rows to design groups.}
}
}

\section{plot}{


\strong{Usage:}

 \code{mg$plot(region_names = NULL, design_names = NULL,
               title = NA, x_label = NA, facet_by=NA, group_by=NA)}

\strong{Description:}

This method produces a ggplot object giving a graphical representation
of the metagene analysis.

\strong{Parameters:}
\describe{
   \item{region_names}{The names of the regions to be plotted. If \code{NULL},
                       all the regions are plotted. Default: \code{NULL}.}
   \item{design_names}{The names of the design groups to be plotted. If 
                       \code{NULL}, all the design groups are
                       plotted. Default: \code{NULL}.}
   \item{title}{A title to add to the graph. \code{NA} can be used to keep 
                the previous value. Default: \code{NA}}
   \item{x_label}{X-axis label for the metagene plot. \code{NA} can be 
                  used to keep the previous value. Default: \code{NA}.}
   \item{facet_by}{A formula to be used for facetting the metagene plot. This
                   formula can include any design metadata, or region_metadata 
                   \code{NA} can be used to keep the previous value.
                   Default: \code{NA}.}
   \item{group_by}{A string representing a single column from design_metadata or region_metadata
                   which will be used to group observations together into lines and which will
                   be used to generate the color scale.
                   \code{NA} can be used to keep the previous value.
                   Default: \code{NA}.}
}
}

\section{Getter methods}{

The following methods return various informations about the metagene object.

\strong{mg$get_params()}
\describe{
   \item{}{Returns a list of all parameters used to perform this metagene analysis.}
}

\strong{mg$get_design()}
\describe{
   \item{}{Returns the design used to perform this metagene analysis.}
}

\strong{mg$get_regions()}
\describe{
   \item{}{Returns the regions used for this metagene analysis.}
}

\strong{mg$get_data_frame(region_names = NULL, design_names = NULL)}
\describe{
   \item{}{Returns full data-frame of results.}
   \item{region_names}{The names of the regions to extract. If \code{NULL},
                       all the regions are returned. Default: \code{NULL}.}
   \item{design_names}{The names of the design groups to extract. If \code{NULL},
                       design groups are returned. Default: \code{NULL}.}
}

\strong{mg$get_plot()}
\describe{
   \item{}{Returns the ggplot object generated by the \code{metagene2$plot} function.}
}

\strong{mg$get_raw_coverages()}
\describe{
   \item{}{Returns raw coverages over the regions specified on initialization.}
}

\strong{mg$get_normalized_coverages()}
\describe{
   \item{}{Returns normalized coverages over the regions specified on initialization.}
}
}

\examples{
mg <- metagene2$new(regions = get_demo_regions(), bam_files = get_demo_bam_files())
\dontrun{
   mg$plot()
}

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{metagene2$new()}}
\item \href{#method-get_bam_count}{\code{metagene2$get_bam_count()}}
\item \href{#method-get_params}{\code{metagene2$get_params()}}
\item \href{#method-get_design}{\code{metagene2$get_design()}}
\item \href{#method-get_design_group_names}{\code{metagene2$get_design_group_names()}}
\item \href{#method-get_regions}{\code{metagene2$get_regions()}}
\item \href{#method-get_regions_metadata}{\code{metagene2$get_regions_metadata()}}
\item \href{#method-get_split_regions}{\code{metagene2$get_split_regions()}}
\item \href{#method-get_data_frame}{\code{metagene2$get_data_frame()}}
\item \href{#method-get_plot}{\code{metagene2$get_plot()}}
\item \href{#method-get_raw_coverages}{\code{metagene2$get_raw_coverages()}}
\item \href{#method-get_normalized_coverages}{\code{metagene2$get_normalized_coverages()}}
\item \href{#method-set_cores}{\code{metagene2$set_cores()}}
\item \href{#method-group_coverages}{\code{metagene2$group_coverages()}}
\item \href{#method-bin_coverages}{\code{metagene2$bin_coverages()}}
\item \href{#method-split_coverages_by_regions}{\code{metagene2$split_coverages_by_regions()}}
\item \href{#method-calculate_ci}{\code{metagene2$calculate_ci()}}
\item \href{#method-add_metadata}{\code{metagene2$add_metadata()}}
\item \href{#method-plot}{\code{metagene2$plot()}}
\item \href{#method-produce_metagene}{\code{metagene2$produce_metagene()}}
\item \href{#method-plot_single_region}{\code{metagene2$plot_single_region()}}
\item \href{#method-replace_region_metadata}{\code{metagene2$replace_region_metadata()}}
\item \href{#method-clone}{\code{metagene2$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$new(
  regions,
  bam_files,
  padding_size = 0,
  cores = SerialParam(),
  verbose = FALSE,
  force_seqlevels = FALSE,
  paired_end = FALSE,
  assay = "chipseq",
  strand_specific = FALSE,
  paired_end_strand_mode = 2,
  region_mode = "auto",
  region_metadata = NULL,
  extend_reads = 0,
  invert_strand = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_bam_count"></a>}}
\if{latex}{\out{\hypertarget{method-get_bam_count}{}}}
\subsection{Method \code{get_bam_count()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_bam_count(filename)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_params"></a>}}
\if{latex}{\out{\hypertarget{method-get_params}{}}}
\subsection{Method \code{get_params()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_params()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_design"></a>}}
\if{latex}{\out{\hypertarget{method-get_design}{}}}
\subsection{Method \code{get_design()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_design()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_design_group_names"></a>}}
\if{latex}{\out{\hypertarget{method-get_design_group_names}{}}}
\subsection{Method \code{get_design_group_names()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_design_group_names()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_regions"></a>}}
\if{latex}{\out{\hypertarget{method-get_regions}{}}}
\subsection{Method \code{get_regions()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_regions()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_regions_metadata"></a>}}
\if{latex}{\out{\hypertarget{method-get_regions_metadata}{}}}
\subsection{Method \code{get_regions_metadata()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_regions_metadata()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_split_regions"></a>}}
\if{latex}{\out{\hypertarget{method-get_split_regions}{}}}
\subsection{Method \code{get_split_regions()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_split_regions()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_data_frame"></a>}}
\if{latex}{\out{\hypertarget{method-get_data_frame}{}}}
\subsection{Method \code{get_data_frame()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_data_frame(region_names = NULL, design_names = NULL)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_plot"></a>}}
\if{latex}{\out{\hypertarget{method-get_plot}{}}}
\subsection{Method \code{get_plot()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_plot()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_raw_coverages"></a>}}
\if{latex}{\out{\hypertarget{method-get_raw_coverages}{}}}
\subsection{Method \code{get_raw_coverages()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_raw_coverages()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_normalized_coverages"></a>}}
\if{latex}{\out{\hypertarget{method-get_normalized_coverages}{}}}
\subsection{Method \code{get_normalized_coverages()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$get_normalized_coverages()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-set_cores"></a>}}
\if{latex}{\out{\hypertarget{method-set_cores}{}}}
\subsection{Method \code{set_cores()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$set_cores(cores)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-group_coverages"></a>}}
\if{latex}{\out{\hypertarget{method-group_coverages}{}}}
\subsection{Method \code{group_coverages()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$group_coverages(
  design = NA,
  normalization = NA,
  design_filter = NA,
  simplify = TRUE
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-bin_coverages"></a>}}
\if{latex}{\out{\hypertarget{method-bin_coverages}{}}}
\subsection{Method \code{bin_coverages()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$bin_coverages(bin_count = NA, region_filter = NA)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-split_coverages_by_regions"></a>}}
\if{latex}{\out{\hypertarget{method-split_coverages_by_regions}{}}}
\subsection{Method \code{split_coverages_by_regions()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$split_coverages_by_regions(split_by = NA)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-calculate_ci"></a>}}
\if{latex}{\out{\hypertarget{method-calculate_ci}{}}}
\subsection{Method \code{calculate_ci()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$calculate_ci(alpha = NA, sample_count = NA, resampling_strategy = NA)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_metadata"></a>}}
\if{latex}{\out{\hypertarget{method-add_metadata}{}}}
\subsection{Method \code{add_metadata()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$add_metadata(design_metadata = NA)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plot"></a>}}
\if{latex}{\out{\hypertarget{method-plot}{}}}
\subsection{Method \code{plot()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$plot(
  region_names = NULL,
  design_names = NULL,
  title = NA,
  x_label = NA,
  facet_by = NA,
  group_by = NA
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-produce_metagene"></a>}}
\if{latex}{\out{\hypertarget{method-produce_metagene}{}}}
\subsection{Method \code{produce_metagene()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$produce_metagene(...)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plot_single_region"></a>}}
\if{latex}{\out{\hypertarget{method-plot_single_region}{}}}
\subsection{Method \code{plot_single_region()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$plot_single_region(
  region,
  facet_by = NA,
  group_by = NA,
  no_binning = FALSE
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-replace_region_metadata"></a>}}
\if{latex}{\out{\hypertarget{method-replace_region_metadata}{}}}
\subsection{Method \code{replace_region_metadata()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$replace_region_metadata(region_metadata)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metagene2$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
