% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groupedWilkinson.R
\name{groupedWilkinson}
\alias{groupedWilkinson}
\title{Combine grouped p-values with Wilkinson's method}
\usage{
groupedWilkinson(p.values, grouping, log.p = FALSE, min.n = 1, min.prop = 0.5)
}
\arguments{
\item{p.values}{A numeric vector containing p-values for individual tests.}

\item{grouping}{A vector or factor of length equal to \code{p.values}, specifying the group to which each test is assigned.

Alternatively, an \link{rle} object where each run corresponds to a group and specifies the entries of \code{p.values} belonging to that group.
This assumes that \code{p.values} is ordered such that all entries in the same group are adjacent to each other.}

\item{log.p}{Logical scalar indicating whether the p-values in \code{p.values} are log-transformed.}

\item{min.n}{Integer scalar specifying the minimum number of individual nulls to reject when testing the joint null.}

\item{min.prop}{Numeric scalar in [0, 1], specifying the minimum proportion of individual nulls to reject when testing the joint null.}
}
\value{
A list containing:
\itemize{
\item \code{p.value}, a named numeric vector of length equal to the number of unique levels in \code{grouping}.
This contains the combined p-value for each group, log-transformed if \code{log.p=TRUE}.
Each entry is named according to the group.
\item \code{representative}, a named integer vector containing the indices of the representative tests for each group.
Each index refers to an entry of \code{p.values} and is named according to its group.
\item \code{influential}, a logical vector of length equal to \code{p.values}.
Entries are \code{TRUE} for any p-value that is deemed \dQuote{influential} to the final combined p-value for its group.
}
}
\description{
Combine p-values from grouped tests with Wilkinson's method.
Groups are defined according to unique levels of a grouping factor.
}
\details{
Here, the joint null hypothesis for each group is all individual null hypotheses are false.
Rejection of the joint null is heavily favored in situations where \eqn{N} or more individual nulls are rejected.
This is achieved in Wilkinson's method by considering the \eqn{N}-th order statistic for uniformly distributed p-values.
The individual tests are assumed to be independent, and all weights are ignored.

\eqn{N} is defined as the larger of \code{min.n} and the product of \code{min.prop} with the number of tests in the group (rounded up).
This allows users to scale rejection of the joint null with the size of the group, while avoiding a too-low \eqn{N} when the group is small.
Note that \eqn{N} is always capped at the total size of the group.

The representative test for each group is defined as that with the \eqn{N}-th smallest p-value, as this is used to compute the combined p-value.
The influential tests for each group are defined as those with p-values no greater than the representative test's p-value.
This is based on the fact that increasing them (e.g., by setting them to unity) would result in a larger combined p-value.
}
\examples{
p1 <- rbeta(100, 0.8, 1)
g <- sample(10, length(p1), replace=TRUE)

# Standard application:
out <- groupedWilkinson(p1, g)
str(out)

# With log p-values. 
out <- groupedWilkinson(log(p1), g, log.p=TRUE)
str(out)

}
\references{
Wilkinson B (1951).
A statistical consideration in psychological research.
\emph{Psychol. Bull.} 48, 156-158.
}
\seealso{
\code{\link{parallelWilkinson}}, for a version that operates on parallel vectors of p-values.

\code{\link{groupedHolmMin}}, which has a more strict interpretation of \emph{N}, amongst other things.
}
\author{
Aaron Lun
}
