#' A container for geneset definitions.
#'
#' Please refer to the sparrow vignette (\code{vignette("sparrow")}),
#' (and the "The GeneSetDb Class" section, in particular) for a more deatiled
#' description of the sematnics of this central data object.
#'
#' The functionality in the class is useful for the functionality in this
#' package, but for your own personal usage, you probably want a `{BiocSet}`.
#'
#' @name GeneSetDb-class
#' @docType class
#' @exportClass GeneSetDb
#'
#' @slot table The "gene set table": a data.table with geneset information,
#'   one row per gene set. Columns include collection, name, N, and n. The end
#'   user can add more columns to this data.table as desired. The actual
#'   feature IDs are computed on the fly by doing a `db[J(group, id)]`
#' @slot db A `data.table` to hold all of the original geneset id
#'   information that was used to construct this `GeneSetDb`.
#' @slot featureIdMap Maps the ids used in the geneset lists to the ids
#'   (rows) over the expression data the GSEA is run on
#' @slot collectionMetadata A `data.table` to keep metadata about each
#'   individual geneset collection, ie. the user might want to keep track of
#'   where the geneset definitions come from. Perhaps a function that parses
#'   the collection,name combination to generate an URL that points the user
#'   to more information about that geneset, etc.
#' @rdname GeneSetDb-class
.GeneSetDb <- setClass("GeneSetDb",
  slots=c(
    table="data.table",
    db="data.table",
    featureIdMap="data.table",
    collectionMetadata="data.table"),
  prototype=prototype(
    table=data.table(
      collection=character(),
      name=character(),
      active=logical(),
      N=integer(),
      n=integer(),
      key=c('collection', 'name')),
    db=data.table(
      collection=character(),
      name=character(),
      feature_id=character(),
      key=c('collection', 'name', 'feature_id')),
    featureIdMap=data.table(
      feature_id=character(),  ## The ids used in the defined genesets
      x.id=character(),
      x.idx=integer(),
      key='feature_id'),
    collectionMetadata=data.table(
      collection=character(),
      name=character(),
      value=list(),
      key=c('collection', 'name'))))

.SparrowRegistry <- setClass("SparrowRegistry",
  slots=c(
    methods="character",
    validate.fn="list",
    do.fn="list"),
  prototype=prototype(
    methods=character(),
    validate.fn=list(),
    do.fn=list()))

#' A SparrowResult object holds the results from a sparrow::seas() call.
#'
#' @description A call to [seas()] will produce analyses for an
#' arbitrary number of GSEA methods, the results of which will be stored and
#' accessible here using the [result()], [results()], and [resultNames()].
#'
#' In addition, the [GeneSetDb()] used for the analysis is accessible
#' via [geneSetDb()], and the results from the differential
#' expression analysis is available via [logFC()].
#'
#' Visualizing results of a geneset based analysis also are functions that
#' operate over a `SparrowResult` object, for instance see the
#' [iplot()] and the `sparrow.shiny` package.
#'
#' @exportClass SparrowResult
#' @rdname SparrowResult
#' @aliases SparrowResult
#'
#' @slot gsd The [GeneSetDb()] this analysis was run over
#' @slot results The list of individual results generated by each of the
#'   GSEA methods that were run.
#' @slot logFC The differential expression statistics for each individual
#'   feature measured in the experiment.
.SparrowResult <- setClass("SparrowResult",
  slots=c(
    gsd="GeneSetDb",
    results="list",
    logFC='data.table'),
  prototype=prototype(
    gsd=new("GeneSetDb"),
    results=list(),
    logFC=data.table(
      feature_id=character(),
      logFC=numeric(),
      t=numeric(),
      pval=numeric(),
      padj=numeric())))
