% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seas.R
\name{seas}
\alias{seas}
\title{Performs a plethora of set enrichment analyses over varied inputs.}
\usage{
seas(
  x,
  gsd,
  methods = NULL,
  design = NULL,
  contrast = NULL,
  use.treat = FALSE,
  feature.min.logFC = if (use.treat) log2(1.25) else 1,
  feature.max.padj = 0.1,
  trim = 0.1,
  verbose = FALSE,
  ...,
  score.by = c("t", "logFC", "pval"),
  rank_by = NULL,
  rank_order = c("ordered", "descending", "ascending"),
  xmeta. = NULL,
  BPPARAM = BiocParallel::SerialParam()
)
}
\arguments{
\item{x}{An object to run enrichment analyses over. This can be an
ExpressoinSet-like object that you can differential expression over
(for roast, fry, camera), a named (by feature_id) vector of scores to run
ranked-based GSEA, a data.frame with feature_id's, ranks, scores, etc.}

\item{gsd}{The \code{\link[=GeneSetDb]{GeneSetDb()}} that defines the gene sets of interest.}

\item{methods}{A character vector indicating the GSEA methods you want to
run. Refer to the GSEA Methods section for more details.
If no methods are specified, only differential gene expression and geneset
level statistics for the contrast are computed.}

\item{design}{A design matrix for the study}

\item{contrast}{The contrast of interest to analyze. This can be a column
name of \code{design}, or a contrast vector which performs "coefficient
arithmetic" over the columns of \code{design}. The \code{design} and \code{contrast}
parameters are interpreted in \emph{exactly} the same way as the same parameters
in limma's \code{\link[limma:camera]{limma::camera()}} and \code{\link[limma:roast]{limma::roast()}} methods.}

\item{use.treat}{should we use limma/edgeR's "treat" functionality for the
gene-level differential expression analysis?}

\item{feature.min.logFC}{The minimum logFC required for an individual
feature (not geneset) to be considered differentialy expressed. Used in
conjunction with \code{feature.max.padj} primarily for summarization
of genesets (by \code{\link[=geneSetsStats]{geneSetsStats()}}, but can also be used by GSEA methods
that require differential expression calls at the individual feature level,
like \code{\link[=goseq]{goseq()}}.}

\item{feature.max.padj}{The maximum adjusted pvalue used to consider an
individual feature (not geneset) to be differentially expressed. Used in
conjunction with \code{feature.min.logFC}.}

\item{trim}{The amount to trim when calculated trimmed \code{t} and
\code{logFC} statistics for each geneset. This is passed down to the
\code{\link[=geneSetsStats]{geneSetsStats()}} function.}

\item{verbose}{make some noise during execution?}

\item{...}{The arguments are passed down into
\code{\link[=calculateIndividualLogFC]{calculateIndividualLogFC()}} and the various geneset analysis functions.}

\item{score.by}{This tells us how to rank the features after differential
expression analysis when \code{x} is an expression container. It specifies the
name of the column to use downstream of a differential expression analysis
over \code{x}. If \code{x} is a data.frame that needs to be ranked, see \code{rank_by}.}

\item{rank_by}{Only works when \code{x} is a data.frame-like input. The name of a
column that should be used to rank the features in \code{x} for pre-ranked gsea
tests like cameraPR or fgsea.  \code{rank_by} overrides \code{score.by}}

\item{rank_order}{Only used when \code{x} is a data.frame-like input. Specifies
how the features in \code{x} should be used to rank the features in \code{x} using
the \code{rank_by} column. Accepted values are:
\code{"ordered"} (default) means that the rows in \code{x} are pre-ranked already.
\code{"descendeing"}, and \code{"ascending"}.}

\item{xmeta.}{A hack to support data.frame inputs for \code{x}. End users should
not use this.}

\item{BPPARAM}{a \emph{BiocParallel} parameter definition, like one generated from
\code{\link[BiocParallel:MulticoreParam-class]{BiocParallel::MulticoreParam()}}, or \code{\link[BiocParallel:BatchtoolsParam-class]{BiocParallel::BatchtoolsParam()}},
for instance, which is passed down to \link{BiocParallel}::bplapply()]. Default
is set to \code{\link[BiocParallel:SerialParam-class]{BiocParallel::SerialParam()}}}
}
\value{
A \code{\link[=SparrowResult]{SparrowResult()}} which holds the results of all the analyses
specified in the \code{methods} parameter.
}
\description{
This is a wrapper function that delegates GSEA analyses to different
"workers", each of which implements the flavor of GSEA of your choosing.
The particular analyses that are performed are specified by the
\code{methods} argument, and these methods are fine tuned by passing their
arguments down through the \code{...} of this wrapper function.
}
\details{
Set enrichment analyses can either be performed over an expression object,
which requires the specification of the experiment design and contrast of
interest, or over a set of features to rank (stored as a data.frame or
vector).

Note that we are currently in the middle of a refactor to accept and fully
take advantage of \code{data.frame} as inputs for \code{x}, which will be used for
preranked type of GSEA methods. See the following issue for more details:
https://github.com/lianos/multiGSEA/issues/24

The bulk of the GSEA methods currently available in this package come from
edgeR/limma, however others are included (and are being added), as well.
\emph{GSEA Methods} and \emph{GSEA Method Parameterization} sections for more details.

In addition to performing GSEA, this function also internally orchestrates
a differential expression analysis, which can be tweaked by identifying
the parameters in the \code{\link[=calculateIndividualLogFC]{calculateIndividualLogFC()}} function, and
passing them down through \code{...} here. The results of the differential
expression analysis (ie. the \code{\link[limma:toptable]{limma::topTable()}}) are accessible by calling
the \code{\link[=logFC]{logFC()}} function on the \code{\link[=SparrowResult]{SparrowResult()}} object returned from this
function call.

\strong{Please Note}: be sure to cite the original GSEA method when using
results generated from this function.
}
\section{GSEA Methods}{

You can choose the methods you would like to run by providing a character
vector of GSEA method names to the \code{methods} parameter. Valid methods
you can select from include:
\itemize{
\item \code{"camera"}: from \code{\link[limma:camera]{limma::camera()}} (*)
\item \code{"cameraPR"}: from \code{\link[limma:camera]{limma::cameraPR()}}
\item \code{"ora"}: overrepresentation analysis optionally accounting for bias
(\code{\link[=ora]{ora()}}). This method is a wrapper function that makes the functionality
in \code{\link[limma:goana]{limma::kegga()}} available to an arbitrary GeneSetDb.
\item \code{"roast"}: from \code{\link[limma:roast]{limma::roast()}} (*)
\item \code{"fry"}: from \code{\link[limma:roast]{limma::fry()}} (*)
\item \code{"romer"}: from \code{\link[limma:romer]{limma::romer()}} (*)
\item \code{"geneSetTest"}: from \code{\link[limma:geneSetTest]{limma::geneSetTest()}}
\item \code{"goseq"}: from \code{\link[goseq:goseq]{goseq::goseq()}}
\item \code{"fgsea"}: from \code{\link[fgsea:fgsea]{fgsea::fgsea()}}
}

Methods annotated with a \verb{(*)} indicate that these methods require
a complete expression object, a valid design matrix, and a contrast
specification in order to run. These are all of the same things you need to
provide when performing a vanilla differential gene expression analysis.

Methods missing a \verb{(*)} can be run on a feature-named input vector
of gene level statistics which will be used for ranking (ie. a named vector
of logFC's or t-statistics for genes). They can also be run by providing
an expression, design, and contrast vector, and the appropriate statistics
vector will be generated internally from the t-statistics, p-values, or
log-fold-changes, depending on the value provided in the \code{score.by}
parameter.

The worker functions that execute these GSEA methods are functions named
\code{do.METHOD} within this package. These functions are not meant to be executed
directly by the user, and are therefore not exported. Look at the respective
method's help page (ie. if you are running \code{"camera"}, look at the
\code{\link[limma:camera]{limma::camera()}} help page for full details. The formal parameters that
these methods take can be passed to them via the \code{...} in this \code{seas()}
function.
}

\section{GSEA Method Parameterization}{


Each GSEA method can be tweaked via a custom set of parameters. We leave the
documentation of these parameters and how they affect their respective GSEA
methods to the documentation available in the packages where they are
defined. The \code{seas()} call simply has to pass these parameters down
into the \code{...} parameters here. The \code{seas} function will then pass these
along to their worker functions.

\strong{What happens when two different GSEA methods have parameters with the
same name?}

Unfortunately you currently cannot provide different values for these
parameters. An upcoming version version of sparrow will support this
feature via slightly different calling semantics. This will also allow the
caller to call the same GSEA method with different parameterizations so that
even these can be compared against each other.
}

\section{Differential Gene Expression}{


When the \code{seas()} call is given an expression matrix, design, and
contrast, it will also internally orchestrate a gene level differential
expression analysis. Depending on the type of expression object passed in
via \code{x}, this function will guess on the best method to use for this
analysis.

If \code{x} is a \code{DGEList}, then ensure that you have already called
\code{\link[edgeR:estimateDisp]{edgeR::estimateDisp()}} on \code{x} and edgeR's quasilikelihood framework will be
used, otherwise we'll use limma (note that \code{x} can be an \code{EList} run through
\code{voom()}, \code{voomWithQuailityWeights()}, or when where you have leveraged
limma's \code{\link[limma:dupcor]{limma::duplicateCorrelation()}} functionality, even.

The parameters of this differential expression analysis can also be
customized. Please refer to the \code{\link[=calculateIndividualLogFC]{calculateIndividualLogFC()}} function for
more information. The \code{use.treat}, \code{feature.min.logFC},
\code{feature.max.padj}, as well as the \code{...} parameters from this function are
passed down to that funciton.
}

\examples{
vm <- exampleExpressionSet()
gdb <- exampleGeneSetDb()
mg <- seas(vm, gdb, c('camera', 'fry'),
           design = vm$design, contrast = 'tumor',
           # customzie camera parameter:
           inter.gene.cor = 0.04)
resultNames(mg)
res.camera <- result(mg, 'camera')
res.fry <- result(mg, 'fry')
res.all <- results(mg)
}
