% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transmogrify.R
\name{transmogrify}
\alias{transmogrify}
\alias{transmogrify,XStringSet,GRanges,GRanges-method}
\alias{transmogrify-methods}
\alias{transmogrify,BSgenome,GRanges,GRanges-method}
\alias{transmogrify,BSgenome,VcfFile,GRanges-method}
\alias{transmogrify,XStringSet,VcfFile,GRanges-method}
\title{Mogrify a transcriptome using a set of variants}
\usage{
transmogrify(x, var, exons, ...)

\S4method{transmogrify}{XStringSet,GRanges,GRanges}(
  x,
  var,
  exons,
  alt_col = "ALT",
  trans_col = "transcript_id",
  omit_ranges = NULL,
  tag = NULL,
  sep = "_",
  var_tags = FALSE,
  var_sep = "_",
  ol_vars = "fail",
  verbose = TRUE,
  mc.cores = 1,
  ...
)

\S4method{transmogrify}{BSgenome,GRanges,GRanges}(
  x,
  var,
  exons,
  alt_col = "ALT",
  trans_col = "transcript_id",
  omit_ranges = NULL,
  tag = NULL,
  sep = "_",
  var_tags = FALSE,
  var_sep = "_",
  ol_vars = "fail",
  verbose = TRUE,
  mc.cores = 1,
  ...
)

\S4method{transmogrify}{BSgenome,VcfFile,GRanges}(
  x,
  var,
  exons,
  alt_col = "ALT",
  trans_col = "transcript_id",
  omit_ranges = NULL,
  tag = NULL,
  sep = "_",
  var_tags = FALSE,
  var_sep = "_",
  ol_vars = "fail",
  verbose = TRUE,
  mc.cores = 1,
  which,
  ...
)

\S4method{transmogrify}{XStringSet,VcfFile,GRanges}(
  x,
  var,
  exons,
  alt_col = "ALT",
  trans_col = "transcript_id",
  omit_ranges = NULL,
  tag = NULL,
  sep = "_",
  var_tags = FALSE,
  var_sep = "_",
  ol_vars = "fail",
  verbose = TRUE,
  mc.cores = 1,
  which,
  ...
)
}
\arguments{
\item{x}{Reference genome as either a DNAStringSet or BSgenome}

\item{var}{GRanges object containing the variants}

\item{exons}{GRanges object with ranges representing exons}

\item{...}{Passed to \link[parallel:mclapply]{parallel::mclapply}}

\item{alt_col}{Column from \code{var} containing alternate bases}

\item{trans_col}{Column from 'exons' containing the transcript_id}

\item{omit_ranges}{GRanges object containing ranges to omit, such as PAR-Y
regions, for example}

\item{tag}{Optional tag to add to all sequence names which were modified}

\item{sep}{Separator to place between seqnames names & tag}

\item{var_tags}{logical(1) Add tags indicating which type of variant were
incorporated, with 's', 'i' and 'd' representing SNPs, Insertions and
Deletions respectively}

\item{var_sep}{Separator between any previous tags and variant tags}

\item{ol_vars}{Error handling for any overlapping variants. See
\link{cleanVariants} for possible values and an explanation}

\item{verbose}{logical(1) Include informative messages, or operate silently}

\item{mc.cores}{Number of cores to be used when multi-threading via
\link[parallel:mclapply]{parallel::mclapply}}

\item{which}{GRanges object passed to \link[VariantAnnotation:ScanVcfParam-class]{VariantAnnotation::ScanVcfParam} if
using a VCF directly}
}
\value{
An XStringSet
}
\description{
Use a set of SNPs, insertions and deletions to modify a reference
transcriptome
}
\details{
Produce a set of variant modified transcript sequences from a standard
reference genome.
Supported variants are SNPs, Insertions and Deletions

Ranges needing to be masked, such as the Y-chromosome, or Y-PAR can be
provided.

\strong{It should be noted that this is a time consuming process}
Inclusion of a large set of insertions and deletions across an entire
transcriptome can involve individually modifying many thousands of
transcripts, which can be a computationally demanding task.
Whilst this can be parallelised using an appropriate number of cores, this
may also prove taxing for lower power laptops, and pre-emptively closing
memory hungry programs such as Slack, or internet browers may be prudent.
}
\examples{
library(GenomicRanges)
library(GenomicFeatures)
seq <- DNAStringSet(c(chr1 = "ACGTAAATGG"))
exons <- GRanges(c("chr1:1-3:-", "chr1:7-9:-"))
exons$transcript_id <- c("trans1")

# When using extractTranscriptSeqs -stranded exons need to be sorted by end
exons <- sort(exons, decreasing = TRUE, by = ~end)
exons
trByExon <- splitAsList(exons, exons$transcript_id)

# Check the sequences
seq
extractTranscriptSeqs(seq, trByExon)

# Define some variants
var <- GRanges(c("chr1:2", "chr1:8"))
var$ALT <- c("A", "GGG")

# Include the variants adding tags to indicate a SNP and indel
# The exons GRanges object will be split by transcript internally
transmogrify(seq, var, exons, var_tags = TRUE)


}
