## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL ## Related to https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE----------------
## Bib setup
library("RefManageR")

## Write bibliography information
bib <- c(
    R = citation(),
    AnnotationHub = citation("AnnotationHub")[1],
    BiocFileCache = citation("BiocFileCache")[1],
    dplyr = citation("dplyr")[1],
    ExperimentHub = citation("ExperimentHub")[1],
    ggplot2 = citation("ggplot2")[1],
    graphics = citation("graphics")[1],
    grDevices = citation("grDevices")[1],
    MatrixGenerics = citation("MatrixGenerics")[1],
    DelayedMatrixStats = citation("DelayedMatrixStats")[1],
    methods = citation("methods")[1],
    purrr = citation("purrr")[1],
    rafalib = citation("rafalib")[1],
    RColorBrewer = citation("RColorBrewer")[1],
    reshape2 = citation("reshape2")[1],
    S4Vectors = citation("S4Vectors")[1],
    scran = citation("scran")[1],
    SingleCellExperiment = citation("SingleCellExperiment")[1],
    spatialLIBD = citation("spatialLIBD")[1],
    stats = citation("stats")[1],
    stringr = citation("stringr")[1],
    SummarizedExperiment = citation("SummarizedExperiment")[1],
    tibble = citation("tibble")[1],
    utils = citation("utils")[1],
    Biobase = citation("Biobase")[1],
    BiocStyle = citation("BiocStyle")[1],
    covr = citation("covr")[1],
    HDF5Array = citation("HDF5Array")[1],
    knitr = citation("knitr")[1],
    RefManageR = citation("RefManageR")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    tidyr = citation("tidyr")[1],
    tidyverse = citation("tidyverse")[1],
    DeconvoBuddies = citation("DeconvoBuddies")[1],
    DeconvoBuddiespaper = citation("DeconvoBuddies")[2]
)

## ----"install", eval = FALSE--------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#     install.packages("BiocManager")
# }
# 
# BiocManager::install("DeconvoBuddies")
# 
# ## Check that you have a valid Bioconductor installation
# BiocManager::valid()

## ----"citation"---------------------------------------------------------------
## Citation info
citation("DeconvoBuddies")

## ----"load packages", message=FALSE, warning=FALSE----------------------------
suppressMessages({
    library("DeconvoBuddies")
    library("SummarizedExperiment")
    library("dplyr")
    library("tidyr")
    library("tibble")
})

## ----`access data-------------------------------------------------------------
## Access and snRNA-seq example data
if (!exists("sce_DLPFC_example")) sce_DLPFC_example <- fetch_deconvo_data("sce_DLPFC_example")

## Explore snRNA-seq data in sce_DLPFC_example
sce_DLPFC_example

## Access Bulk RNA-seq data
if (!exists("rse_gene")) rse_gene <- fetch_deconvo_data("rse_gene")

## Explore bulk data in rse_gene
rse_gene

## ----`get_mean_ratio demo`----------------------------------------------------
## find marker genes with get_mean_ratio
marker_stats <- get_mean_ratio(
    sce_DLPFC_example,
    cellType_col = "cellType_broad_hc",
    gene_name = "gene_name",
    gene_ensembl = "gene_id"
)

## explore tibble output, gene with high MeanRatio values are good marker genes
marker_stats

## ----`create_cell_colors demo 1`----------------------------------------------
test_cell_types <- c("cell_A", "cell_B", "cell_C", "cell_D", "cell_E")

## Preview "classic" colors
test_cell_colors_classic <- create_cell_colors(
    cell_types = test_cell_types,
    palette_name = "classic",
    preview = TRUE
)

## Preview "gg" colors
test_cell_colors_gg <- create_cell_colors(
    cell_types = test_cell_types,
    palette_name = "gg",
    preview = TRUE
)

## Preview "tableau" colors
test_cell_colors_tableau <- create_cell_colors(
    cell_types = test_cell_types,
    palette_name = "tableau",
    preview = TRUE
)

## Check the color hex codes for "tableau"
test_cell_colors_tableau

## Provide a palette from RColorBrewer
test_cell_colors_brew <- create_cell_colors(
    cell_types = test_cell_types,
    palette = RColorBrewer::brewer.pal(n = length(test_cell_types), name = "Dark2"),
    preview = TRUE
)

## ----create_cell_colors demo 2`-----------------------------------------------
my_cell_types <- levels(sce_DLPFC_example$cellType_hc)
## Ignore any suffix after the "_" character by using the "split" argument
my_cell_colors <- create_cell_colors(
    cell_types = my_cell_types,
    palette_name = "classic",
    preview = TRUE,
    split = "_"
)

## ----`plot_marker_expression demo`--------------------------------------------
# plot expression of the top 6 Astro marker genes
plot_marker_express(
    sce = sce_DLPFC_example,
    stats = marker_stats,
    cell_type = "Astro",
    n_genes = 6,
    cellType_col = "cellType_broad_hc",
    color_pal = my_cell_colors
)

## ----`demo plot_composition_bar`----------------------------------------------
# load example data
data("rse_bulk_test")
data("est_prop_test")

# access the colData of a test rse dataset
pd <- colData(rse_bulk_test) |>
    as.data.frame()

## pivot data to long format and join with test estimated proportion data
est_prop_long <- est_prop_test |>
    rownames_to_column("RNum") |>
    pivot_longer(!RNum, names_to = "cell_type", values_to = "prop") |>
    left_join(pd)

## explore est_prop_long
est_prop_long

## the composition bar plot shows cell type composition for Sample
plot_composition_bar(est_prop_long,
    x_col = "RNum",
    add_text = FALSE
) +
    ggplot2::scale_fill_manual(values = test_cell_colors_classic)

## the composition bar plot shows the average cell type composition for each Dx
plot_composition_bar(est_prop_long, x_col = "Dx") +
    ggplot2::scale_fill_manual(values = test_cell_colors_classic)

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----vignetteBiblio, results = "asis", echo = FALSE, warning = FALSE, message = FALSE---------------------------------
## Print bibliography
PrintBibliography(bib, .opts = list(hyperlink = "to.doc", style = "html"))

