## ----style, echo=FALSE, results='hide', message=FALSE-------------------------
BiocStyle::markdown()

## ----package_install, eval=FALSE, message=FALSE, warning=FALSE----------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# if (!require("MDSvis", quietly = TRUE))
#     BiocManager::install("MDSvis")

## ----rlibs, results=FALSE, message=FALSE, warning=FALSE-----------------------
library(HDCytoData)
library(CytoMDS)
library(MDSvis)

## ----loadDataSet--------------------------------------------------------------
Krieg_fs <- Krieg_Anti_PD_1_flowSet()
Krieg_fs

## ----convertPhenoData---------------------------------------------------------
# update phenoData structure

chLabels <- 
    keyword(Krieg_fs[[1]], "MARKER_INFO")$MARKER_INFO$channel_name
chMarkers <- 
    keyword(Krieg_fs[[1]], "MARKER_INFO")$MARKER_INFO$marker_name
marker_class <- 
    keyword(Krieg_fs[[1]], "MARKER_INFO")$MARKER_INFO$marker_class

chLabels <- chLabels[marker_class != "none"]
chMarkers <- chMarkers[marker_class != "none"]
# marker_class all equal to "type", 24 markers are left

phenoData <- flowCore::pData(Krieg_fs)
additionalPhenoData <- 
    keyword(Krieg_fs[[1]], "EXPERIMENT_INFO")$EXPERIMENT_INFO
phenoData <- cbind(phenoData, additionalPhenoData)
    
flowCore::pData(Krieg_fs) <- phenoData

## ----distCalc-----------------------------------------------------------------
# transform flow set (arcsinh(cofactor = 5))
trans <- arcsinhTransform(
    transformationId="ArcsinhTransform", 
    a = 0, 
    b = 1/5, 
    c = 0)

# Applying arcsinh() transformation
Krieg_fs_trans <- transform(
    Krieg_fs,
    transformList(chLabels, trans))

# Calculating Sample distances
pwDist <- pairwiseEMDDist(
    Krieg_fs_trans,
    channels = chMarkers,
    verbose = FALSE
)

## ----MDSCalc------------------------------------------------------------------
mdsObj <- computeMetricMDS(
    pwDist,
    seed = 0)

show(mdsObj)

## ----statsCalc----------------------------------------------------------------
# Computing sample statistics

statFUNs <- c(
    "median" = stats::median, 
    "std-dev" = stats::sd,
    "mean" = base::mean,
    "Q25" = function(x, na.rm) 
        stats::quantile(x, probs = 0.25, na.rm = na.rm),
    "Q75" = function(x, na.rm) 
        stats::quantile(x, probs = 0.75, na.rm = na.rm))

chStats <- CytoMDS::channelSummaryStats(
    Krieg_fs_trans,
    channels = chMarkers,
    statFUNs = statFUNs)


## ----saveRDS, eval=FALSE------------------------------------------------------
# saveRDS(mdsObj, file = "Krieg_mdsObj.rds")
# saveRDS(phenoData, file = "Krieg_phenoData.rds")
# saveRDS(chStats, file = "Krieg_chStats.rds")

## ----launch, eval = FALSE-----------------------------------------------------
# MDSvis::mdsvis_app()
# 
# # alternatively launch the application in demo mode, with the Krieg_Anti_PD_1
# # dataset already loaded.
# #MDSvis::mdsvis_app(preLoadDemoDataset = TRUE)
# 

## ----sessioninfo, echo=FALSE--------------------------------------------------
sessionInfo()

