#' Format elapsed time between two timestamps
#'
#' @description
#' Computes the elapsed time between a start and end timestamp, returning 
#' a list with minutes and seconds. If the duration is less than one 
#' minute, only seconds are returned. Useful for logging runtimes in a 
#' human-readable format.
#'
#' @param end_time POSIXct. The end time of the process.
#' @param start_time POSIXct. The start time of the process.
#' @param units Character. Units for \code{difftime()} calculation. 
#'     Default is \code{"secs"}.
#'
#' @return A named list containing:
#' \itemize{
#'     \item \code{mins} (optional) — integer number of minutes
#'     \item \code{secs} — integer number of seconds
#' }
#'
#' @keywords internal
#' @examples
#' \dontrun{
#'     start <- Sys.time()
#'     Sys.sleep(3.5)
#'     end <- Sys.time()
#'     runtime(end, start)
#' }
#' 
runtime <- function(end_time, start_time, units = "secs") {
    total_seconds <- as.numeric(
        difftime(end_time, start_time, units = units)
    )

    mins <- floor(total_seconds / 60)
    secs <- total_seconds - (mins * 60)

    if (mins > 0) {
        return(list(mins = mins, secs = secs))
    } else {
        return(list(secs = secs))
    }
}


#' Extract Model Results into a Unified Data Frame
#'
#' @description
#' Extracts scalar results from a named list of model objects
#' from `glmmTMB` fits into a tidy data frame. Handles nested lists, 
#' missing or `NULL` models, and ensures consistent columns across 
#' all entries.
#'
#' @param sumExp A SummarizedExperiment object containing fitted model
#' objects stored in `rowData(sumExp)[['DOUResults']]`.
#'
#' @param verbose Logical. If \code{TRUE}, messages will be printed
#' for skipped or failed models.
#'
#' @return A data frame where each row corresponds to an ORF and its
#' associated model results. Columns include scalar parameters
#' extracted from the model, \code{orf_id}, and \code{model_type}.
#'
#' @details
#' The function uses a recursive helper (`flatten_scalars`) to extract
#' scalar values from nested lists. It supports model type
#' \code{"glmmTMB"}, and gracefully handles \code{NULL} or unsupported 
#' model types by returning minimal rows.
#'
#' Missing columns across models are filled with \code{NA} to ensure a
#' consistent structure.
#'
#' @import SummarizedExperiment
#' @importFrom stats setNames
#' 
#' @keywords internal
#' 
extract_results <- function(sumExp, verbose = TRUE) {
    models_list <- rowData(sumExp)[["DOUResults"]]
    
    flatten_scalars <- function(x, prefix = NULL) {
        if (is.atomic(x) && length(x) == 1) {
            name <- if (is.null(prefix)) "value" else prefix
            return(setNames(list(x), name))
        } else if (is.list(x)) {
            result <- list()
            list_names <- names(x)
            if (is.null(list_names)) {
                list_names <- paste0("elem", seq_along(x))
            }
            for (i in seq_along(x)) {
                n <- list_names[i]
                sub_prefix <- if (is.null(prefix)) n else paste0(prefix, ".", n)
                result <- c(result, flatten_scalars(x[[i]], sub_prefix))
            }
            return(result)
        } else {
            return(list())
        }
    }
    
    process_model <- function(model_obj, name) {
        if (is.null(model_obj)) {
            if (verbose) {
                message(
                    "Skipping orf_id ", 
                    name, 
                    " due to NULL model object."
                )
            }
            return(data.frame(
                orf_id = name,
                model_type = "NULL",
                stringsAsFactors = FALSE
            ))
        }
        
        model_type <- model_obj@type
        
        if (model_type == "glmmTMB") {
            params <- model_obj@results
            param_values <- flatten_scalars(params)
            param_values$orf_id <- name
            param_values$model_type <- model_type
            df <- as.data.frame(param_values, stringsAsFactors = FALSE)
        } else {
            df <- data.frame(
                orf_id = name,
                model_type = model_type,
                stringsAsFactors = FALSE
            )
        }
        
        return(df)
    }
    
    results_df_list <- lapply(names(models_list), function(name) {
        process_model(models_list[[name]], name)
    })
    
    all_cols <- unique(unlist(lapply(results_df_list, names)))
    
    results_df_list_filled <- lapply(results_df_list, function(df) {
        missing <- setdiff(all_cols, names(df))
        for (col in missing) df[[col]] <- NA
        df[all_cols]
    })
    
    results_df <- do.call(rbind, results_df_list_filled)
    rownames(results_df) <- NULL
    
    return(results_df)
}
