% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posthoc.R
\name{testDOU}
\alias{testDOU}
\title{Compute Differential ORF Usage (DOU) Contrasts Using EMMs}
\usage{
testDOU(
  data,
  contrasts_method = "revpairwise",
  nullweight = 500,
  verbose = TRUE
)
}
\arguments{
\item{data}{A \code{\link{DOUData-class}} object containing \code{emmGrid}
objects, typically stored in \code{rowData(data)[['DOUResults']]}.}

\item{contrasts_method}{Character string specifying the method
for computing contrasts. Default is \code{"revpairwise"}.}

\item{nullweight}{Numeric. Prior weight on the null hypothesis
for empirical Bayes shrinkage. Higher values yield more
conservative lfsr estimates. Default is \code{500}.}

\item{verbose}{Logical. If \code{TRUE}, prints progress messages.
Default is \code{TRUE}.}
}
\value{
A \code{\link{DOUData-class}} object with two new
\code{S4Vectors::DataFrame} slots. These tables contain
long-format results for all computed contrasts:
\describe{
\item{\code{interaction}}{
A long-format \code{S4Vectors::DataFrame} containing DOU
effect sizes (log-odds of Ribo-seq minus RNA-seq) for all
interaction-specific contrasts. Columns include:
\code{contrast}, \code{betahat} (raw log-odds effect size),
\code{sebetahat} (standard error), \code{waldpval}
(Wald test p-value), \code{waldpadj}
(adjusted p-value), \code{posterior} (posterior mean from
shrinkage), and \code{lfsr} (Local False Sign Rate).
}
\item{\code{strategy}}{
A long-format \code{S4Vectors::DataFrame} containing
strategy-specific effect sizes (e.g., Ribo-seq only) for all
computed contrasts. Columns include:
\code{strategy}, \code{contrast}, and the same shrunken and
unshrunken metrics as above.
}
}
}
\description{
Performs Differential ORF Usage (DOU) analysis by computing
contrasts between ribosome profiling and RNA-seq modalities
using estimated marginal means (EMMs) from fitted models.
Supports interaction-specific and strategy-specific contrasts.
Applies empirical Bayes shrinkage via \code{\link[ashr]{ash}} to
stabilize effect size estimates.
}
\details{
Results for post hoc contrasts are stored in long format using
explicit \code{contrast} and/or \code{strategy} columns.
Non-converged models are omitted.
}
\examples{
# Load SummarizedExperiment to enable subsetting and access to 
# components like rowRanges and rowData
library(SummarizedExperiment)

# Read in count matrix, condition table, and annotation files
dir <- system.file("extdata", package = "DOTSeq")

cnt <- read.table(
    file.path(dir, "featureCounts.cell_cycle_subset.txt.gz"),
    header = TRUE,
    comment.char = "#"
)
names(cnt) <- gsub(".*(SRR[0-9]+).*", "\\\\1", names(cnt))

gtf <- file.path(dir, "gencode.v47.orf_flattened_subset.gtf.gz")
bed <- file.path(dir, "gencode.v47.orf_flattened_subset.bed.gz")

meta <- read.table(file.path(dir, "metadata.txt.gz"))
names(meta) <- c("run", "strategy", "replicate", "treatment", "condition")
# extract only samples processed using cyclohexamide
cond <- meta[meta$treatment == "chx", ]
cond$treatment <- NULL # remove the treatment column

# Create a DOTSeqDataSets object
d <- DOTSeqDataSetsFromFeatureCounts(
    count_table = cnt,
    condition_table = cond,
    flattened_gtf = gtf,
    flattened_bed = bed
)

# Keep ORFs where all replicates in at least one condition pass min_count
# Single-ORF genes are removed
dou <- getDOU(d)
dou <- dou[rowRanges(dou)$is_kept == TRUE, ]

# Randomly sample 100 ORFs for fitDOU
set.seed(42)
random_rows <- sample(seq_len(nrow(dou)), size = 100)
dou <- dou[random_rows, ]

# Model fitting using fitDOU
rowData(dou)[["DOUResults"]] <- fitDOU(
    data = dou,
    formula = ~ condition * strategy,
    specs = ~ condition * strategy,
    dispersion_modeling = "auto",
    lrt = FALSE,
    optimizers = FALSE,
    diagnostic = FALSE,
    parallel = list(n = 4L, autopar = TRUE),
    verbose = TRUE
)

# Run post hoc contrasts, Wald tests, and effect size shrinkage
dou <- testDOU(dou, verbose = TRUE)

}
\references{
Lenth R, Piaskowski J (2025). emmeans: Estimated Marginal Means, aka
Least-Squares Means. R package version 2.0.0,
\url{https://rvlenth.github.io/emmeans/}

Stephens, M. (2016) False discovery rates: a new deal.
Biostatistics, 18:2. DOI: 10.1093/biostatistics/kxw041
}
\seealso{
\code{\link{DOTSeq}}, \code{\link{DOTSeqDataSets-class}},
\code{\link{fitDOU}}, \code{\link{plotDOT}}
}
