#' EventPointer RNASeq IGV Visualization
#'
#' Generates of files to be loaded in IGV for visualization and interpretation of events
#'
#' @param Events Data.frame generated by EventPointer_RNASeq with the events to be included in the GTF file.
#' @param SG_RNASeq Output from PrepareBam_EP function. Contains splicing graphs components.
#' @param EventsTxt Path to EventsFound.txt file generated with EventDetection function
#' @param PathGTF Directory where to write the GTF files.
#'
#'
#' @return The function displays a progress bar to show the user the progress of the function. Once the progress bar reaches 100%, one .gtf
#' file is written to the specified directory in PathGTF. The created file: 1) paths_RNASeq.gtf : GTF file representing the alternative splicing
#' events.
#'
#' @examples
#' \dontrun{
#'   data(AllEvents_RNASeq)
#'   data(SG_RNASeq)
#'
#'    # Run EventPointer
#'
#'    Dmatrix<-matrix(c(1,1,1,1,1,1,1,1,0,0,0,0,1,1,1,1),ncol=2,byrow=FALSE)
#'    Cmatrix<-t(t(c(0,1)))
#'    Events <- EventPointer_RNASeq(AllEvents_RNASeq,Dmatrix,Cmatrix,Statistic='LogFC',PSI=TRUE)
#'
#'    # IGV Visualization
#'
#'    EventsTxt<-paste(system.file('extdata',package='EventPointer'),'/EventsFound_RNASeq.txt',sep='')
#'    PathGTF<-tempdir()
#'    EventPointer_RNASeq_IGV(Events,SG_RNASeq,EventsTxt,PathGTF)
#'    }
#' @export


EventPointer_RNASeq_IGV <- function(Events, 
    SG_RNASeq, EventsTxt, PathGTF) {
    if (is.null(Events)) {
        stop("Missing alternative splicing events")
    }
    
    if (is.null(SG_RNASeq)) {
        stop("Missing splicing graphs information")
    }
    
    # if (is.null(EventsTxt) | classsss(EventsTxt) 
    #  != "character") {
    if (is.null(EventsTxt) | 
        !is(EventsTxt,"character")) {
        stop("Wrong or missing EventsTxt field")
    }
    
    # if (is.null(PathGTF) | 
    #  classsss(PathGTF) != "character") {
    if (is.null(PathGTF) | 
        !is(PathGTF,"character")) {
        stop("Wrong or missing PathGTF field")
    }
    
    
    
    SgF <- rowRanges(SG_RNASeq)
    
    ######################## iiP<-which(strand(SgF)@values=='+')
    ######################## iiN<-which(strand(SgF)@values=='-')
    ######################## strand(SgF)@values[iiP]<-'-'
    ######################## strand(SgF)@values[iiN]<-'+'
    
    # Create file to store gtf for patths
    # (events)
    FILE.paths <- paste(PathGTF, "/paths_RNASeq.gtf", 
        sep = "")
    cat(file = FILE.paths, paste("#track name=", 
        shQuote("paths", type = "cmd"), " gffTags=", 
        shQuote("on", type = "cmd"), sep = ""), 
        "\n")
    
    # Read EventsFound txt
    EventsInfo <- read.delim(file = EventsTxt, 
        sep = "\t", header = TRUE, stringsAsFactors = FALSE)
    # xx<-paste(EventsInfo[,1],'_',EventsInfo[,2],sep='')
    # rownames(EventsInfo)<-xx
    
    cat("\n Generating GTF Files...")
    
    pb <- txtProgressBar(min = 0, max = nrow(Events), 
        style = 3)
    
    for (jj in seq_len(nrow(Events))) {
        setTxtProgressBar(pb, jj)
        
        ### 3 Hay que ajustar esta l?nea
        ### Gene<-unlist(strsplit(Events[jj,1],'_'))[1]
        EventXX <- as.numeric(unlist(strsplit(rownames(Events)[jj], 
            "_")))
        Gene <- EventXX[1]
        EvNumb <- EventXX[2]
        # Gene<-EventsInfo[jj,1]
        
        SG_Gene <- SgF[unlist(geneID(SgF)) == 
            Gene]
        
        SG_Edges <- SG_Info(SG_Gene)$Edges
        
        if (unique(SG_Edges[, "Strand"]) == 
            "") {
            SG_Edges[, "Strand"] <- "-"
        }
        
        # iixx<-match(Events[jj,1],rownames(EventsInfo))
        iixx <- match(rownames(Events)[jj], 
            EventsInfo[, 1])
        
        EventPaths <- GetIGVPaths(EventsInfo[iixx, 
            ], SG_Edges)
        class(EventPaths[, 2]) <- "integer"
        class(EventPaths[, 3]) <- "integer"
        WriteGTF_RNASeq(PathGTF, EventsInfo[iixx, 
            ], EventPaths)
    }
    
    
    close(pb)
    cat("\n")
}
