#' @title Compute the Matrix of Jaccard Indexes
#'
#' @description This function computes the Jaccard index for each pair of
#' nodes contained in TDAobj, generated by the  \code{\link{mapperCore}}
#' function. The resulting data.frame can be used to represent data as a
#' network, for instance, in Cytoscape
#'
#' @param x A TDAobj object, processed by the  \code{\link{mapperCore}}
#' function.
#' @return The starting TDAobj object, in which the matrix of Jaccard indexes,
#' calculated comparing each node of the 'dfMapper' slot, has been added
#'  (slot: 'jacc')
#'
#' @details The Jaccard index measures the similarity of two nodes A and B.
#' It ranges from 0 to 1. If A and B share no members, their Jaccard index
#' would be 0 (= NA). If A and B share all members, their Jaccard index would
#' be 1. Hence, the higher the index, the more similar the
#' two nodes. If the Jaccard index between A and B is different from NA, it
#' means that an edge exists between A and B. The output matrix of Jaccard
#' indexes can be used as an adjacency matrix. The resulting data.frame can
#' be used to represent data as a network, for instance, in Cytoscape.
#'
#' @author Mattia Chiesa, Laura Ballarini, Luca Piacentini
#'
#' @examples
#' ## use example data:
#' data(tda_test_data)
#' jacc_mat <- jaccardMatrix(tda_test_data)
#'
#' @seealso
#' \code{\link{makeTDAobj}},
#' \code{\link{dfToDistance}},
#' \code{\link{dfToProjection}},
#' \code{\link{mapperCore}}
#'
#' @export
#'
#'
jaccardMatrix <- function (x) {

  # checks-------------------------------
  if (!is(x,'TDAobj'))
    stop("'x' argument must be a TDAobj object")

  dfMapper <- getDfMapper(x)

  # check missing arguments
  if (missing(dfMapper))
    stop("'dfMapper' argument must be provided")

  # check the type of argument
  if (!is.data.frame(dfMapper))
    stop("'dfMapper' argument must be a data.frame")

  # specific checks
  if (ncol(dfMapper) != 1)
    stop("ncol(dfMapper) must be equal to 1")

  if (!(all(vapply(dfMapper, is.character,logical(1)))))
    stop("'dfMapper' variables must be numeric")

  # check the presence of NA or Inf
  if (any(is.na(dfMapper)))
    stop("NA values are not allowed in the 'dfMapper' data.frame")

  if (any(is.infinite(as.matrix(dfMapper))))
    stop("Inf values are not allowed in the 'dfMapper' data.frame")

  # body----------
  jaccDataRes <- matrix(nrow = nrow(dfMapper), ncol=nrow(dfMapper))
  rownames(jaccDataRes) <- rownames(dfMapper)
  colnames(jaccDataRes) <- rownames(dfMapper)
  # computation of the returned matrix (i.e. 'jaccDataRes')
  for (i in seq_len(nrow(dfMapper))) {
    for (j in seq_len(nrow(dfMapper))) {

      listDfRowElem1 <- unlist(strsplit(dfMapper[i, 1], " "))
      listDfRowElem2 <- unlist(strsplit(dfMapper[j, 1], " "))

      if(jaccard(listDfRowElem1, listDfRowElem2) != 0) {
        jaccDataRes[i, j] <- jaccard(listDfRowElem1, listDfRowElem2)
        jaccDataRes[j, i] <- jaccard(listDfRowElem1, listDfRowElem2)
      }
    }
  }

  diag(jaccDataRes) <- NA

  x <- setJacc(x, jaccDataRes)

  return(x)
}
