% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decemedip.R
\name{decemedip}
\alias{decemedip}
\title{Main function to perform cell type deconvolution with MeDIP-seq data}
\usage{
decemedip(
  sample_bam_file = NULL,
  paired_end = NULL,
  counts_cts = c(),
  counts_anc = c(),
  ref_assembly = "hg19",
  ref_cts = NULL,
  ref_anc = NULL,
  weight_cts = 1,
  weight_anc = 0.5,
  diagnostics = TRUE,
  seed = 2024,
  cores = 1,
  chains = 4,
  iter = 2000,
  stan_input_params = list(s_mu = 3, s_sigma = 3, n_knot_z = 0, degree_z = 3, Xi =
    cor(as.matrix(SummarizedExperiment::assay((ref_cts)))), s_theta = 3, s_tau = 3),
  stan_control = list(adapt_delta = 0.95, max_treedepth = 15),
  timeout_sec = 2 * (diagnostics + 1) * iter * chains,
  max_retries = 3,
  ...
)
}
\arguments{
\item{sample_bam_file}{A string value that indicates the file path to the bam file of a
MeDIP-seq sample of interest. If \code{sample_bam_file} is specified, please do not
specify \code{counts_cts} and \code{counts_anc} to avoid conflict.}

\item{paired_end}{A logic value that indicates whether the bam file is from paired-end
reads or single-end. Specify \verb{TRUE\} for paired-end and \code\{FALSE} for single-end.}

\item{counts_cts}{An atomic vector of integer values that indicates the read counts of
a MeDIP-seq sample on reference sites/regions. If \code{counts_cts} and \code{counts_anc}
is specified, please do not specify \code{sample_bam_file} and \code{paired_end} to avoid conflict.}

\item{counts_anc}{An atomic vector of integer values that indicates the read counts of
a MeDIP-seq sample on reference sites/regions. If  \code{counts_cts} and \code{counts_anc}
is specified, please do not specify \code{sample_bam_file} and \code{paired_end} to avoid conflict.}

\item{ref_assembly}{A string that represents the genome assembly that should be used for cell type-specific
sites in the reference panel ('hg19' or 'hg38'). Default to 'hg19'. The default reference is explained
in the manuscript of \pkg{decemedip}. Alternatively, if the user want to provide their own
reference panel by using the \code{ref_cts} and \code{ref_anc} arguments.}

\item{ref_cts}{A \code{SummarizedExperiment} object that contains the genomic coordinates and
beta values of the cell type-specific sites/regions from reference cell types.
The \verb{\link{makeReferencePanel}} can be used to generate such a panel.}

\item{ref_anc}{Same as \code{ref_cts} but for anchor sites.}

\item{weight_cts}{A numeric value indicating the weights that should be put on cell type-specific
sites/regions. Default is 0.5.}

\item{weight_anc}{A numeric value indicating the weights that should be put on cell type-specific
sites/regions. Default is 1.}

\item{diagnostics}{A logic value that indicates whether to include components of the stan model
in the output that are necessary for future diagnostics of the model, such as posterior
predictive checks. For details, please refer to the function \verb{\link{plotDiagnostics}}.}

\item{seed}{The seed for random number generation in MCMC sampling.}

\item{cores}{A positive integer specifying the number of cores that can be used for
MCMC sampling. The default is 1.}

\item{chains}{A positive integer specifying the number of Markov chains. The default is 4.}

\item{iter}{A positive integer specifying the number of iterations for each chain (including
warmup). The default is 2000.}

\item{stan_input_params}{A named list of parameters that specifies the prior of the
decemedip model.}

\item{stan_control}{A named list of parameters to control the sampler's behavior in
Stan. See the details in the documentation for the control argument in \verb{\link[rstan]\{stan\}}.}

\item{timeout_sec}{A numerical value indicating the CPU/processor time (in seconds) allowed for
the MCMC to run before restarting the chains with a new random seed.}

\item{max_retries}{An integer value indicating the maximum number of tries with different seed
for MCMC if it fails to converge.}

\item{...}{Other parameters that can be passed to the \verb{\link[rstan]\{sampling\}} function.}
}
\value{
A list of two elements:
\enumerate{
\item \code{data_list}: An organized list of variables used as input to the Stan posterior sampling function.
\item \code{posterior}: An \code{stanfit} object produced by Stan representing the fitted posteriors.
}
}
\description{
Main function to perform cell type deconvolution with MeDIP-seq data
}
\examples{

# Here we use a lightweighted example that only contains 8 cell types to avoid long running time:

data(example.hg19.ref.cts.se)
data(example.hg19.ref.anc.se)
data(example.pdx.counts.cts.se)
data(example.pdx.counts.anc.se)

# read counts of cell type-specific CpGs of the sample 'LuCaP_147CR'
counts_cts <- SummarizedExperiment::assays(example.pdx.counts.cts.se)$counts[, "LuCaP_147CR"]
# read counts of anchor CpGs of the sample 'LuCaP_147CR'
counts_anc <- SummarizedExperiment::assays(example.pdx.counts.anc.se)$counts[, "LuCaP_147CR"]

## Fit decemedip model
output <- decemedip(counts_cts = counts_cts, counts_anc = counts_anc,
                    ref_cts = example.hg19.ref.cts.se, ref_anc = example.hg19.ref.anc.se,
                    iter = 500, cores = 1, chains = 1)

## IMPORTANT NOTE (PLEASE READ): For usual cases, you may skip specifying ref_cts and ref_anc,
## as the default panels contain 25 cell types, as used in the manuscript, e.g., run the function as:
\dontrun{
  ## By default, the functions uses 2000 iterations, 4 cores and 4 chains
  output <- decemedip(counts_cts = counts_cts, counts_anc = counts_anc)
  ## or use BAM files as input (paired=TRUE is file is paired-end)
  output <- decemedip(sample_bam_file = "path/to/bam/files", paired = TRUE)
}

}
