#' Plot a glycan structure represented as an igraph tree
#'
#' @description
#' Visualize a glycan structure encoded as an igraph object produced by
#' \code{\link{build_glycan_igraph}}. This plot is mainly intended for inspecting
#' glycan topology (branching, residue types, connectivity).
#'
#' @param g
#' An igraph object representing a glycan tree from \code{\link{build_glycan_igraph}}.
#'
#' @return A glycan topology plot is drawn as a side effect.
#'
#'
#' @examples
#' # Example: parse a pGlyco3-style glycan expression into a tree
#' pGlyco_expr <- "(N(N(H(H(H))(H(H)(H)(H(H))))))"
#'
#' # Convert to glycan tree structure
#' tree <- pGlyco3_to_tree(pGlyco_expr)
#'
#' # Explore parsed nodes and edges
#' tree$node
#' tree$edge
#'
#' # Build igraph representation
#' g <- build_glycan_igraph(tree)
#' plot_glycan_tree(g)
#' @export
plot_glycan_tree <- function(g) {
  plot(
    g,
    layout = igraph::layout_as_tree(g, root = "a"),
    vertex.label = igraph::V(g)$type,
    vertex.color = igraph::V(g)$color,
    vertex.size = 30,
    edge.arrow.size = 1,
    main = "Glycan Topology"
  )
}


#' Plot the distribution of a glycan trait across experimental groups
#'
#' @description
#' Generate two diagnostic plots—a histogram and a boxplot—for a selected
#' glycan trait at a specific site/protein feature.
#' This function extracts GPSM-level values from a trait matrix stored in a
#' `SummarizedExperiment`, subsets them by group, removes missing values,
#' and visualizes the resulting distribution.
#'
#' @param trait_se
#' A SummarizedExperiment object generated by \link{build_trait_se},
#' containing one assay matrix per glycan trait.
#'
#' @param group_col
#' Column name in `colData(trait_se)` that defines sample group membership.
#'
#' @param group_levels
#' Character vector specifying the group values to include in the plot.
#' Typically two values (e.g. `c("Control","Treatment")`).
#'
#' @param trait_name
#' Name of the glycan trait to plot. Must match an assay name in
#' `assays(trait_se)`.
#'
#' @param feature
#' Row identifier within the trait matrix (peptide, or protein).
#' Must match a row name in `assays(trait_se)[[trait_name]]`.
#'
#' @return
#' A named list of two `ggplot2` objects:
#' * `p_hist` — histogram of trait intensities
#' * `p_box`  — boxplot with jitter overlay
#'
#'
#' @examples
#' # Load the toy GPSM table exported by pGlyco3 (included in the package)
#' path <- system.file("extdata", "pGlyco3_gpsm_toyexample.txt",
#'   package = "glycoTraitR"
#' )
#' gpsm_toyexample <- read_pGlyco3_gpsm(path)
#'
#' # Load the accompanying toy metadata
#' data("meta_toyexample")
#'
#' # Build a protein-level glycan trait SummarizedExperiment object
#' trait_se <- build_trait_se(
#'   gpsm = gpsm_toyexample,
#'   from = "pGlyco3",
#'   motifs = NULL,
#'   level = "protein",
#'   meta = meta_toyexample
#' )
#'
#' # Identify glycan traits significantly changed between two groups
#' changed_traits <- analyze_trait_changes(
#'   trait_se     = trait_se,
#'   group_col    = "Diagnosis",
#'   group_levels = c("Normal", "Symptomatic"),
#'   min_psm      = 20
#' )
#'
#' # Extract one trait name and one protein/site feature to visualize
#' trait_name <- changed_traits$trait[1]
#' feature <- changed_traits$level[1]
#'
#' # Plot the distribution of this selected trait
#' p <- plot_trait_distribution(
#'   trait_se     = trait_se,
#'   group_col    = "Diagnosis",
#'   group_levels = c("Normal", "Symptomatic"),
#'   trait_name   = trait_name,
#'   feature      = feature
#' )
#'
#' # Show histogram and boxplot
#' p$p_hist
#' p$p_box
#'
#' @export
plot_trait_distribution <- function(trait_se,
                                    group_col,
                                    group_levels,
                                    trait_name,
                                    feature) {
  assays <- SummarizedExperiment::assays(trait_se)
  coldata <- SummarizedExperiment::colData(trait_se)
  rowdata <- SummarizedExperiment::rowData(trait_se)

  # extract abundance vector for the specified trait and feature row
  abund <- assays[[trait_name]][feature, ]
  label <- coldata[[group_col]]

  # subset: only PSMs belonging to selected group levels
  col_ind <- label %in% group_levels
  label_new <- label[col_ind]
  abund_new <- abund[col_ind]

  # keep non-NA values
  ind <- !is.na(abund_new)
  label_new <- label_new[ind]
  abund_new <- abund_new[ind]

  # build plotting data
  df_plot <- data.frame(
    Abundance = as.numeric(abund_new),
    Group     = factor(label_new, levels = group_levels)
  )

  ## freq hist
  p_freq <-
    ggplot2::ggplot(
      data = df_plot,
      ggplot2::aes(x = .data$Abundance, fill = .data$Group)
    ) +
    ggplot2::geom_histogram(
      ggplot2::aes(y = ggplot2::after_stat(count)),
      position = "dodge",
      color = "black",
      binwidth = max(1, ceiling(max(df_plot$Abundance) / 20))
    ) +
    ggplot2::scale_fill_manual(
      values = stats::setNames(c("#e31a1c", "#1f78b4"), group_levels)
    ) +
    ggplot2::labs(
      x = paste(trait_name, "Count at", feature),
      y = "PSM count"
    ) +
    ggplot2::theme_classic(base_size = 14) +
    ggplot2::theme(
      plot.title = ggplot2::element_text(hjust = 0.5, face = "bold"),
      axis.text.x = ggplot2::element_text(size = 14),
      axis.text.y = ggplot2::element_text(size = 12),
      legend.position = "top",
      legend.title = ggplot2::element_blank(),
      legend.text = ggplot2::element_text(size = 14)
    )

  ## boxplot
  p_box <-
    ggplot2::ggplot(
      data = df_plot,
      ggplot2::aes(
        x = .data$Group,
        y = .data$Abundance,
        fill = .data$Group
      )
    ) +
    ggplot2::geom_boxplot(width = 0.5, show.legend = FALSE) +
    ggplot2::geom_jitter(size = 1, width = 0.1, show.legend = FALSE) +
    ggplot2::scale_fill_manual(
      values = stats::setNames(c("#e31a1c", "#1f78b4"), group_levels)
    ) +
    ggplot2::labs(
      x = NULL,
      y = paste(trait_name, "Count at", feature)
    ) +
    ggplot2::theme_classic(base_size = 14) +
    ggplot2::theme(
      axis.text.x = ggplot2::element_text(size = 14),
      axis.text.y = ggplot2::element_text(size = 12),
      legend.position = "top",
      legend.title = ggplot2::element_blank(),
      legend.text = ggplot2::element_text(size = 14)
    )

  return(list(p_hist = p_freq, p_box = p_box))
}
