% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/export.R
\name{exportMiXCR}
\alias{exportMiXCR}
\title{Export Reference Sequences to MiXCR Format}
\usage{
exportMiXCR(
  sequences,
  output_dir,
  chain = c("IGH", "IGK", "IGL", "TRA", "TRB", "TRD", "TRG")
)
}
\arguments{
\item{sequences}{A \code{\link[Biostrings]{DNAStringSet}} or
\code{\link[Biostrings]{AAStringSet}} object containing immune receptor
sequences. Sequence names must follow standard IG/TR gene nomenclature
(e.g., \code{"IGHV1-2*01"}, \code{"TRBJ2-1*01"}). Can be obtained from
\code{\link{getIMGT}} or \code{\link{getOGRDB}}.}

\item{output_dir}{Character string specifying the directory where output
files will be written. The directory will be created if it does not exist.}

\item{chain}{Character string specifying the chain type for the output files.
Must be one of \code{"IGH"}, \code{"IGK"}, \code{"IGL"}, \code{"TRA"},
\code{"TRB"}, \code{"TRD"}, or \code{"TRG"}.}
}
\value{
A named list containing the paths to the created files, returned
  invisibly. The list may contain elements \code{v_genes}, \code{d_genes},
  \code{j_genes}, and \code{c_genes} depending on which segment types were
  found in the input sequences.
}
\description{
Exports a \code{\link[Biostrings]{DNAStringSet}} or
\code{\link[Biostrings]{AAStringSet}} to FASTA files formatted for use with
MiXCR's \code{buildLibrary} command. The function creates separate FASTA
files for V, D, J, and C gene segments.
}
\details{
MiXCR expects FASTA files with simple headers containing only the gene name.
The function filters sequences by gene type (V, D, J, C) based on the gene
name pattern and writes separate files for each segment type.

Output files follow the naming convention:
\itemize{
  \item \code{v-genes.<chain>.fasta}
  \item \code{d-genes.<chain>.fasta}
  \item \code{j-genes.<chain>.fasta}
  \item \code{c-genes.<chain>.fasta}
}

This function works with sequences from both \strong{IMGT} (via
\code{\link{getIMGT}}) and \strong{OGRDB} (via \code{\link{getOGRDB}}).
}
\examples{
# Create a small example DNAStringSet
seqs <- Biostrings::DNAStringSet(c(
  "ATGCGATCGATCGATCG",
  "ATGCGATCGATCG",
  "ATGCGATC",
  "ATGCGATCGATCGATCGATCG"
))
names(seqs) <- c("IGHV1-2*01", "IGHD1-1*01", "IGHJ1*01", "IGHC*01")

# Export to temporary directory
output_dir <- tempdir()
files <- exportMiXCR(seqs, output_dir, chain = "IGH")
print(files)

# Clean up
unlink(unlist(files))
}
\seealso{
\code{\link{getIMGT}}, \code{\link{getOGRDB}} for obtaining sequences

\code{\link{exportTRUST4}}, \code{\link{exportCellRanger}},
\code{\link{exportIgBLAST}} for other export formats

\url{https://mixcr.com/mixcr/guides/create-custom-library/} for MiXCR
documentation
}
