% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tracks.R
\name{track_hic_matrix}
\alias{track_hic_matrix}
\title{Generate HiC track}
\usage{
track_hic_matrix(
  file,
  title = NULL,
  height = NULL,
  overlay_previous = "no",
  orientation = NULL,
  max_value = NULL,
  min_value = NULL,
  transform = "no",
  rasterize = TRUE,
  colormap = "RdYlBu_r",
  depth = 100000,
  show_masked_bins = FALSE,
  scale_factor = 1
)
}
\arguments{
\item{file}{String. The location of the track file}

\item{title}{String. If specificed, the title of the track to be displayed.}

\item{height}{Numeric. The height of the plotted track in cm. Default is 2.
See notes.}

\item{overlay_previous}{String. Options are "no" (default) or "yes" or
"share-y".}

\item{orientation}{String. Set to "inverted" to make the track upside down.
Default is NULL.}

\item{max_value}{Numeric. Default is NULL. The max value cut-off for the
numeric column.}

\item{min_value}{Numeric. Default is NULL. The max value cut-off for the
numeric column.}

\item{transform}{String to indicate type of transformation applied.
Default is "no".}

\item{rasterize}{Boolean. Default is FALSE.}

\item{colormap}{String with matplotlib-compatible colormap. Default is set to "viridis".}

\item{depth}{Numeric value above 1 to indicate the maximum distance that should be plotted. Default is 100000.}

\item{show_masked_bins}{Boolean. If TRUE, showing masked bins as white lines. Default is FALSE.}

\item{scale_factor}{Numeric factor by which matrix is to be scaled.}
}
\value{
genom_track
}
\description{
Create a genome_track for matrix files. Currently, only cool format and h5 format.
}
\details{
This function expect cool or h5 format. Format converter like
\href{https://hicexplorer.readthedocs.io/en/latest/content/tools/hicConvertFormat.html#hicconvertformat}{hicConvertFormat}
can help converting to supported formats.
depth is the maximum distance that should be plotted.
If it is more than 125\% of the plotted region, it will be adjsted to this maximum value.
colormap argument should be compatible with \href{https://matplotlib.org/2.0.2/users/colormaps.html}{matplotlib}.
show_masked_bins plots bins not used during the corrections as white lines.
Setting this argument to FALSE (default) extends neighboring bins to
obtain an aesthetically pleasant output.
scale argument scales the matrix by specific factor.
This is useful if plotting multiple hic-matrices to be on the same scale.
}
\examples{
\dontrun{
# Get example data directories
# Download h5 example
ah <- AnnotationHub()
query(ah, "rGenomeTracksData")
h5_dir <- ah[["AH95901"]]
tads_dir <- system.file("extdata", "tad_classification.bed",
  package = "rGenomeTracks"
)
arcs_dir <- system.file("extdata", "links2.links", package = "rGenomeTracks")
bw_dir <- system.file("extdata", "bigwig2_X_2.5e6_3.5e6.bw", package = "rGenomeTracks")
#
# Create HiC track from HiC matrix
h5 <- track_hic_matrix(
  file = h5_dir, depth = 250000, min_value = 5, max_value = 200,
  transform = "log1p", show_masked_bins = FALSE
)

# Create TADS track
tads <- track_domains(
  file = tads_dir, border_color = "black",
  color = "none", height = 5,
  line_width = 5,
  show_data_range = FALSE,
  overlay_previous = "share-y"
)

# Create arcs track
arcs <- track_links(
  file = arcs_dir, links_type = "triangles", line_style = "dashed",
  overlay_previous = "share-y",
  color = "darkred",
  line_width = 3,
  show_data_range = FALSE
)

# Create bigwig track
bw <- track_bigwig(
  file = bw_dir, color = "red",
  max_value = 50,
  min_value = 0,
  height = 4,
  overlay_previous = "yes",
  show_data_range = FALSE
)

# Create one object from HiC, arcs and bigwid
tracks <- h5 + arcs + bw

# Plot the tracks
plot_gtracks(tracks, chr = "X", start = 25 * 10^5, end = 31 * 10^5)
# Plot HiC, TADS and bigwig tracks
plot_gtracks(h5 + tads + bw, chr = "X", start = 25 * 10^5, end = 31 * 10^5)
}
}
\author{
Omar Elashkar
}
